package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerVcardStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.extractVcardId
import ru.yandex.grut.objects.proto.BannerV2
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Status.TModerationStatus
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Status.TModerationStatus.TItemStatus
import java.time.Duration

/**
 * Отдельное апи для репликации баннеров, так как в репликации заполняются не все поля
 */
open class BannerReplicationGrutApi(
    grutContext: GrutContext,
    properties: GrutApiProperties
) : BannerGrutApi(grutContext, properties, setPaths = /* unused */ emptyList()) {

    protected companion object {
        const val skipModerationPath = "/status/skip_moderation"
        const val moderationStatusPath = "/status/moderation_status/main_status"
        const val vcardModerationStatusPath = "/status/moderation_status/vcard_status"
        const val imageModerationStatusPath = "/status/moderation_status/images_status"
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }

    private fun serializeSpecProto(obj: BannerGrut): BannerV2.TBannerV2Spec {
        return BannerV2.TBannerV2Spec.newBuilder().apply {
            serializeBannerFields(this, obj.banner)
            serializeForeignBannerFields(this, obj)
        }.build()
    }

    override fun serializeSpec(obj: BannerGrut): ByteString {
        return serializeSpecProto(obj).toByteString()
    }

    override fun createOrUpdateBanners(banners: List<BannerGrut>) {
        val payloads = banners.map { serializeBanner(it) }
        createOrUpdateObjects(payloads)
    }

    override fun createOrUpdateBannersParallel(banners: List<BannerGrut>) {
        val payloads = banners.map { serializeBanner(it) }
        createOrUpdateObjectsParallel(payloads, UPDATE_TIMEOUT)
    }

    /**
     * Модерация баннера может быть включена через процесс над GrUT'ом (skipModeration = false). Если так, то
     * не отправляем moderationStatus. Пока полностью не перейдем на модерацию через GrUT, в Директе баннер будет
     * модерироваться тоже (через ess транспорт).
     */
    private fun serializeStatus(obj: BannerGrut, setRemovePaths: SetRemovePaths): BannerV2.TBannerV2Status {
        setRemovePaths.pathsToSet.add(skipModerationPath)
        return BannerV2.TBannerV2Status.newBuilder().apply {
            skipModeration = obj.skipModeration
            if (obj.skipModeration) {
                moderationStatus = serializeModerationStatus(obj, setRemovePaths)
            }
        }.build()
    }

    private fun serializeModerationStatus(obj: BannerGrut, setRemovePaths: SetRemovePaths): TModerationStatus {
        val builder = TModerationStatus.newBuilder()

        val banner = obj.banner
        builder.mainStatus = serializeMainModerationStatus(banner.statusModerate)
        setRemovePaths.pathsToSet.add(moderationStatusPath)

        if (banner is BannerWithVcard
            && banner.vcardStatusModerate != null
            && extractVcardId(banner) != null
            && !obj.dropVcard
        ) {
            val vcardVerdict = when (banner.vcardStatusModerate) {
                BannerVcardStatusModerate.YES -> TModerationStatus.EVerdict.MV_ACCEPTED.number
                BannerVcardStatusModerate.NO -> TModerationStatus.EVerdict.MV_REJECTED.number
                else -> TModerationStatus.EVerdict.MV_UNKNOWN.number
            }
            builder.vcardStatus = TItemStatus.newBuilder().apply { verdict = vcardVerdict }.build()
            setRemovePaths.pathsToSet.add(vcardModerationStatusPath)
        } else {
            setRemovePaths.pathsToRemove.add(vcardModerationStatusPath)
        }

        if (canSendImage(obj)) {
            val imagesVerdict = when ((banner as BannerWithBannerImage).imageStatusModerate) {
                StatusBannerImageModerate.YES -> TModerationStatus.EVerdict.MV_ACCEPTED.number
                StatusBannerImageModerate.NO -> TModerationStatus.EVerdict.MV_REJECTED.number
                else -> TModerationStatus.EVerdict.MV_UNKNOWN.number
            }
            builder.addImagesStatus(TItemStatus.newBuilder().apply { verdict = imagesVerdict }.build())
            setRemovePaths.pathsToSet.add(imageModerationStatusPath)
        } else {
            setRemovePaths.pathsToRemove.add(imageModerationStatusPath)
        }

        return builder.build()
    }

    protected fun serializeMainModerationStatus(statusModerate: BannerStatusModerate): TItemStatus {
        val mainVerdict = when (statusModerate) {
            BannerStatusModerate.YES -> TModerationStatus.EVerdict.MV_ACCEPTED.number
            BannerStatusModerate.NO -> TModerationStatus.EVerdict.MV_REJECTED.number
            else -> TModerationStatus.EVerdict.MV_UNKNOWN.number
        }
        return TItemStatus.newBuilder().apply { verdict = mainVerdict }.build()
    }

    private fun serializeBanner(banner: BannerGrut): UpdatedObject {
        val meta = serializeMeta(banner)
        val spec = serializeSpecProto(banner)

        val setRemove = prepareSetRemovePaths(spec, "/spec", bannerSpecFields.plus(bannerForeignSpecFields))
        val status = serializeStatus(banner, setRemove)

        // всегда реплицируется в meta (значение или 0)
        // обновляем так как картинку могли добавить позже + раньше не реплицировали удаленные картинки
        setRemove.pathsToSet.add("/meta/image_banner_id")

        return UpdatedObject(
            meta = meta,
            spec = spec.toByteString(),
            status = status.toByteString(),
            setPaths = setRemove.pathsToSet,
            removePaths = setRemove.pathsToRemove
        )
    }
}
