package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.bidmodifier.BidModifierABSegmentAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierBannerTypeAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierPrismaIncomeGradeAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRegionalAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierRetargetingFilterAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTabletAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierTrafaretPositionAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifier.BidModifierWeatherAdjustment
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionParameter
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.mysql2grut.enummappers.BidModifierEnumMappers.Mappers.toGrut
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.BidModifier.TBidModifierSpec
import ru.yandex.grut.objects.proto.BidModifier.TBidModifierSpec.Builder
import ru.yandex.grut.objects.proto.BidModifier.TBidModifierSpec.TConjunction
import ru.yandex.grut.objects.proto.BidModifier.TBidModifierSpec.TDisjunction
import ru.yandex.grut.objects.proto.BidModifier.TBidModifierSpec.TExpressionAtom
import ru.yandex.grut.objects.proto.BidModifier.TBidModifierSpec.TGeo
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.client.Schema.TBidModifier
import java.time.Duration

class BidModifierGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<BidModifierGrut>(grutContext, Schema.EObjectType.OT_BID_MODIFIER, properties) {

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TBidModifierMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TBidModifierMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: BidModifierGrut): ByteString {
        val hierarhicalMultiplierType = BidModifierType.toSource(obj.type)!!
        val grutType = toGrut(hierarhicalMultiplierType)
        return Schema.TBidModifierMeta.newBuilder()
            .setId(obj.id)
            .setClientId(obj.clientId)
            .setBidModifierType(grutType.number)
            .build()
            .toByteString()
    }

    // Корректировки, для которых не заполняется выражение
    // (чей typeSupport релизует BidModifierSingleValueTypeSupport)
    private val simpleBidModifierTypes = setOf(
        BidModifierType.DESKTOP_ONLY_MULTIPLIER,
        BidModifierType.DESKTOP_MULTIPLIER,
        BidModifierType.PERFORMANCE_TGO_MULTIPLIER,
        BidModifierType.VIDEO_MULTIPLIER,
        BidModifierType.SMARTTV_MULTIPLIER,
    )

    override fun serializeSpec(obj: BidModifierGrut): ByteString {
        return TBidModifierSpec.newBuilder().apply {
            priceCoefficientPercent = obj.adjustment.percent.toLong()

            when (obj.adjustment) {
                is BidModifierMobileAdjustment -> number = toGrut(obj.adjustment.osType).number.toLong()
                is BidModifierTabletAdjustment -> number = toGrut(obj.adjustment.osType).number.toLong()
                is BidModifierRetargetingAdjustment -> retargetingConditionId = obj.adjustment.retargetingConditionId
                is BidModifierRetargetingFilterAdjustment -> retargetingConditionId = obj.adjustment.retargetingConditionId
                is BidModifierABSegmentAdjustment -> retargetingConditionId = obj.adjustment.abSegmentRetargetingConditionId
                is BidModifierRegionalAdjustment -> geo = TGeo.newBuilder()
                    .setRegionId(obj.adjustment.regionId)
                    .setIsHidden(obj.adjustment.hidden)
                    .build()
                is BidModifierInventoryAdjustment -> number = toGrut(obj.adjustment.inventoryType).number.toLong()
                is BidModifierBannerTypeAdjustment -> number = toGrut(obj.adjustment.bannerType).number.toLong()
                is BidModifierTrafaretPositionAdjustment -> number = toGrut(obj.adjustment.trafaretPosition).number.toLong()
                is BidModifierDemographicsAdjustment -> demography = TBidModifierSpec.TDemography.newBuilder()
                    .setAgeGroup(toGrut(obj.adjustment.age).number)
                    .setGender(toGrut(obj.adjustment.gender).number)
                    .build()
                is BidModifierWeatherAdjustment -> buildWeatherExpression(this, obj.adjustment)
                is BidModifierPrismaIncomeGradeAdjustment -> buildExpression(this, obj.adjustment)
                else ->
                    if (!simpleBidModifierTypes.contains(obj.type)) {
                        throw RuntimeException("Unsupported adjustment, type: ${obj.type}, id: ${obj.adjustment.id}")
                    }
            }

        }
            .build()
            .toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return TBidModifier.parseFrom(rawMeta).meta.id
    }

    fun getBidModifier(id: Long): TBidModifier? {
        return getObjectAs(id, ::transformBidModifier)
    }

    fun getBidModifiers(
        ids: Collection<Long>,
        attributes: List<String> = listOf("/meta", "/spec", "/status")
    ): Collection<TBidModifier> {
        return getObjectsByIds(ids, attributes).map { transformBidModifier(it)!! }
    }


    private val setPaths = listOf("/spec")
    fun createOrUpdateBidModifiers(objects: Collection<BidModifierGrut>) {
        createOrUpdateObjects(objects, setPaths)
    }

    fun createOrUpdateBidModifiersParallel(objects: Collection<BidModifierGrut>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, setPaths = setPaths)
    }

    private fun transformBidModifier(raw: ObjectApiServiceOuterClass.TVersionedPayload?): TBidModifier? {
        if (raw == null) return null
        return TBidModifier.parseFrom(raw.protobuf)
    }

    fun buildExpression(builder: Builder, adjustment: BidModifierExpressionAdjustment) {
        val conjunction = TConjunction.newBuilder()
        for (orExpression in adjustment.condition) {
            val orList = TDisjunction.newBuilder()
            for (expressionAtom in orExpression) {
                val atom = TExpressionAtom.newBuilder().apply {
                    operator = toGrut(expressionAtom.operation).number
                    if (expressionAtom.parameter.equals(BidModifierExpressionParameter.PRISMA_INCOME_GRADE)) {
                        number = Integer.parseInt(expressionAtom.valueString)
                    } else {
                        if (expressionAtom.valueInteger != null) {
                            number = expressionAtom.valueInteger
                        }
                        if (expressionAtom.valueString != null) {
                            throw RuntimeException(
                                "unsupported valueType: ${expressionAtom.valueString}," +
                                    "adjustment: ${adjustment.javaClass}, id: ${adjustment.id}")
                        }
                    }
                }.build()
                orList.addExpressions(atom)
            }
            conjunction.addDisjunctions(orList.build())
        }
        builder.conjunction = conjunction.build()
    }

    fun buildWeatherExpression(builder: Builder, adjustment: BidModifierWeatherAdjustment) {
        val conjunction = TConjunction.newBuilder()
        for (orExpression in adjustment.expression) {
            val orList = TDisjunction.newBuilder()
            for (weatherAtom in orExpression) {
                val atom = TExpressionAtom.newBuilder().apply {
                    operator = toGrut(weatherAtom.operation).number
                    weather = TExpressionAtom.TWeather.newBuilder()
                        .setValue(weatherAtom.value)
                        .setWeatherParameter(toGrut(weatherAtom.parameter).number).build()
                }.build()
                orList.addExpressions(atom)
            }
            conjunction.addDisjunctions(orList.build())
        }
        builder.conjunction = conjunction.build()
    }
}

data class BidModifierGrut(
    val id: Long,
    val clientId: Long,
    val type: BidModifierType,
    val adjustment: BidModifierAdjustment,
)
