package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.toGrutMoney
import ru.yandex.direct.core.mysql2grut.repository.BiddableShowCondition
import ru.yandex.direct.mysql2grut.enummappers.AdGroupEnumMappers
import ru.yandex.direct.mysql2grut.enummappers.BiddableShowConditionEnumMappers
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.BiddableShowCondition.TBiddableShowConditionSpec
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration

class BiddableShowConditionGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<BiddableShowCondition>(grutContext, Schema.EObjectType.OT_BIDDABLE_SHOW_CONDITION, properties) {
    override fun buildIdentity(id: Long): ByteString {
        return Schema.TBiddableShowConditionMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TBiddableShowConditionMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: BiddableShowCondition): ByteString {
        val id = directIdToGrutId(obj)

        return Schema.TBiddableShowConditionMeta.newBuilder()
            .setId(id)
            .setClientId(obj.clientId)
            .build()
            .toByteString()
    }

    private val setPaths = listOf("/spec")
    fun createOrUpdateBiddableShowConditions(objects: Collection<BiddableShowCondition>) {
        createOrUpdateObjects(objects, setPaths = setPaths)
    }

    fun createOrUpdateBiddableShowConditionsParallel(objects: Collection<BiddableShowCondition>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, setPaths = setPaths)
    }


    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TBiddableShowCondition.parseFrom(rawMeta).meta.id
    }

    override fun serializeSpec(obj: BiddableShowCondition): ByteString {
        return TBiddableShowConditionSpec.newBuilder().apply {
            when (obj.type) {
                BiddableShowConditionType.KEYWORD ->
                    keyword = TBiddableShowConditionSpec.TKeyword.newBuilder().apply {
                        price = toGrutMoney(obj.keyword!!.price)
                        priceContext = toGrutMoney(obj.keyword.priceContext)
                        phrase = obj.keyword.phrase
                        isSuspended = obj.keyword.isSuspended
                        if (obj.keyword.hrefParam1 != null) {
                            hrefParam1 = obj.keyword.hrefParam1
                        }
                        if (obj.keyword.hrefParam2 != null) {
                            hrefParam2 = obj.keyword.hrefParam2
                        }
                        if (obj.keyword.showsForecast != null) {
                            showsForecast = obj.keyword.showsForecast
                        }
                    }.build()
                BiddableShowConditionType.OFFER_RETARGETING ->
                    offerRetargeting = TBiddableShowConditionSpec.TOfferRetargeting.newBuilder().apply {
                        isSuspended = obj.offerRetargeting!!.isSuspended
                    }.build()
                BiddableShowConditionType.RELEVANCE_MATCH ->
                    relevanceMatch = TBiddableShowConditionSpec.TRelevanceMatch.newBuilder().apply {
                        price = toGrutMoney(obj.relevanceMatch!!.price)
                        priceContext = toGrutMoney(obj.relevanceMatch.priceContext)
                        isSuspended = obj.relevanceMatch.isSuspended
                        if (obj.relevanceMatch.hrefParam1 != null) {
                            hrefParam1 = obj.relevanceMatch.hrefParam1
                        }
                        if (obj.relevanceMatch.hrefParam2 != null) {
                            hrefParam2 = obj.relevanceMatch.hrefParam2
                        }
                        addAllCategories(obj.relevanceMatch.categories.map { AdGroupEnumMappers.toGrutRelevanceMatchCategory(it).number })
                    }.build()
                BiddableShowConditionType.DYNAMIC ->
                    dynamic = TBiddableShowConditionSpec.TDynamic.newBuilder().apply {
                        price = toGrutMoney(obj.dynamic!!.price)
                        priceContext = toGrutMoney(obj.dynamic.priceContext)
                        isSuspended = obj.dynamic.isSuspended
                        if (obj.dynamic.conditionName != null) {
                            condition = TBiddableShowConditionSpec.TDynamic.TDynamicCondition.newBuilder().apply {
                                name = obj.dynamic.conditionName
                            }.build()
                        }
                    }.build()
                BiddableShowConditionType.PERFORMANCE ->
                    performance = TBiddableShowConditionSpec.TPerformance.newBuilder().apply {
                        priceCpc = toGrutMoney(obj.performance!!.priceCPC)
                        priceCpa = toGrutMoney(obj.performance.priceCPA)
                        name = obj.performance.name
                        targetFunnel = BiddableShowConditionEnumMappers.toGrutTargetFunnel(obj.performance.targetFunnel).number
                    }.build()
                BiddableShowConditionType.RETARGETING -> {
                    retargeting = TBiddableShowConditionSpec.TRetargeting.newBuilder().apply {
                        priceContext = toGrutMoney(obj.retargeting!!.priceContext)
                    }.build()
                    this.retargetingConditionId = obj.retargeting!!.retargetingConditionId
                }

            }

        }.build().toByteString()
    }

    fun getObjectsByDirectIds(type: BiddableShowConditionType, directIds: List<Long>): List<Schema.TBiddableShowCondition> {
        val grutIds = directIds.map { directIdToGrutId(type, it) }
        val identities = grutIds.map { buildIdentity(it) }
        val rawObjects = getObjects(identities)
        return rawObjects.filter { it.protobuf.size() > 0 }
            .map { transformToBiddableShowCondition(it)!! }
    }

    fun getBiddableShowConditions(ids: List<Long>): List<Schema.TBiddableShowCondition> {
        val rawObjects = getObjectsByIds(ids)
        return rawObjects.filter { it.protobuf.size() > 0 }
            .map { transformToBiddableShowCondition(it)!! }
    }


    fun deleteBiddableShowConditionsByDirectIdsParallel(directIdsByType: Map<BiddableShowConditionType, List<Long>>) {
        if (directIdsByType.isEmpty()) return
        val grutIds = mutableListOf<Long>()
        for ((type, ids) in directIdsByType) {
            grutIds.addAll(ids.map { directIdToGrutId(type, it) })
        }
        deleteObjectsParallel(grutIds, ignoreNonexistent = true, DELETE_TIMEOUT)
    }

    fun getExistingBiddableShowConditionsParallel(ids: Collection<Long>): List<Long> {
        return getExistingObjectsParallel(ids, GET_TIMEOUT)
    }

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
        private val DELETE_TIMEOUT = Duration.ofMinutes(1)
        private val GET_TIMEOUT = Duration.ofMinutes(1)


        private fun directIdToGrutId(model: BiddableShowCondition): Long {
            return directIdToGrutId(model.type, model.id)
        }

        fun grutIdToDirectId(type: BiddableShowConditionType, id: Long): Long {
            return when(type) {
                BiddableShowConditionType.KEYWORD,
                BiddableShowConditionType.RELEVANCE_MATCH,
                BiddableShowConditionType.OFFER_RETARGETING -> id
                BiddableShowConditionType.DYNAMIC -> (1L shl 58) xor id          // (1 << 58) ^ id
                BiddableShowConditionType.PERFORMANCE -> (1L shl 59) xor id      // (1 << 59) ^ id
                BiddableShowConditionType.RETARGETING -> (1L shl 60) xor id      // (1 << 60) ^ id
            }

        }


        fun directIdToGrutId(type: BiddableShowConditionType, id: Long): Long {
            return when (type) {
                BiddableShowConditionType.KEYWORD,
                BiddableShowConditionType.RELEVANCE_MATCH,
                BiddableShowConditionType.OFFER_RETARGETING -> id
                BiddableShowConditionType.DYNAMIC -> (1L shl 58) or id          // 1 << 58 | id
                BiddableShowConditionType.PERFORMANCE -> (1L shl 59) or id      // 1 << 59 | id
                BiddableShowConditionType.RETARGETING -> (1L shl 60) or id      // 1 << 60 | id
            }
        }


        private fun transformToBiddableShowCondition(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TBiddableShowCondition? {
            if (raw == null) return null
            return Schema.TBiddableShowCondition.parseFrom(raw.protobuf)
        }
    }
}

enum class BiddableShowConditionType {
    KEYWORD,
    PERFORMANCE,
    RETARGETING,
    RELEVANCE_MATCH,
    OFFER_RETARGETING,
    DYNAMIC,
}
