package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.landing.model.BizLanding
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesFormatsNamespace
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.BizLanding as GrutBizLanding

private val setPaths = listOf("/spec")

private val DIRECT_AVATARS_IMAGE_URL_PARTS = listOf(
    "/get-${BannerImagesFormatsNamespace.direct.literal}/",
    "/get-${BannerImagesFormatsNamespace.direct_picture.literal}/",
)

class BizLandingGrutApi(grutContext: GrutContext, properties: GrutApiProperties) :
        GrutApiBase<BizLanding>(grutContext, Schema.EObjectType.OT_BIZ_LANDING, properties) {

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TBizLandingMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TBizLandingMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: BizLanding): ByteString {
        return Schema.TBizLandingMeta.newBuilder()
                .apply {
                    if (obj.id != null && obj.id > 0) {
                        id = obj.id
                    }
                }
                .build()
                .toByteString()
    }

    override fun serializeSpec(obj: BizLanding): ByteString {
        return GrutBizLanding.TBizLandingSpec.newBuilder()
            .apply {
                url = obj.url

                // Сохраняем только те ссылки, которые лежат в Аватарнице Директа
                if (isDirectAvatarsImageUrl(obj.logoUrl)) {
                    logoImageUrl = obj.logoUrl
                }
                if (isDirectAvatarsImageUrl(obj.coverUrl)) {
                    coverImageUrl = obj.coverUrl
                }
            }
            .build()
            .toByteString()
    }

    private fun isDirectAvatarsImageUrl(url: String?) =
        if (url == null) {
            false
        } else {
            DIRECT_AVATARS_IMAGE_URL_PARTS.any { url.contains(it) }
        }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TBizLanding.parseFrom(rawMeta).meta.id
    }

    fun getBizLanding(id: Long): BizLanding? {
        return getObjectAs(id, ::transformToBizLanding)
    }

    fun createOrUpdateBizLanding(bizLanding: BizLanding) {
        createOrUpdateObject(bizLanding, setPaths)
    }

    private fun transformToBizLanding(raw: ObjectApiServiceOuterClass.TVersionedPayload?): BizLanding? {
        if (raw == null) {
            return null
        }
        val parsed = Schema.TBizLanding.parseFrom(raw.protobuf) ?: return null
        val meta = parsed.meta
        val spec = parsed.spec
        return BizLanding().apply {
            id = meta.id
            url = spec.url
            logoUrl = if (spec.hasLogoImageUrl()) spec.logoImageUrl else null
            coverUrl = if (spec.hasCoverImageUrl()) spec.coverImageUrl else null
        }
    }
}
