package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.AdGroup
import ru.yandex.grut.objects.proto.AdGroup.TAdGroupSpec.EDirectAdGroupStatus
import ru.yandex.grut.objects.proto.client.Schema

data class BriefAdGroup(
    val id: Long,
    val briefId: Long,
    val status: EDirectAdGroupStatus? = null,
)


/**
 * Группа для связки заявки(campaign) и сгенерированного баннера для uc
 */
class BriefAdGroupGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<BriefAdGroup>(grutContext, Schema.EObjectType.OT_AD_GROUP, properties) {
    private val setPaths = listOf("/spec")

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TAdGroupMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TAdGroupMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: BriefAdGroup): ByteString {
        return Schema.TAdGroupMeta.newBuilder().apply {
            id = obj.id
            campaignId = obj.briefId
        }.build().toByteString()
    }

    override fun serializeSpec(obj: BriefAdGroup): ByteString {
        return AdGroup.TAdGroupSpec.newBuilder().apply {
            obj.status?.let { directAdGroupStatus = it }
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TAdGroup.parseFrom(rawMeta).meta.id
    }

    fun selectAdGroups(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec"),
        index: String? = null,
        limit: Long? = null,
        continuationToken: String? = null,
        allowFullScan: Boolean = false
    ): List<Schema.TAdGroup> {
        return selectObjectsAs(filter, attributeSelector, index, limit, continuationToken, allowFullScan, ::transformToBriefAdGroup)
    }

    fun createOrUpdateBriefAdGroups(briefAdGroups: List<BriefAdGroup>) {
        createOrUpdateObjects(briefAdGroups, setPaths)
    }

    fun createOrUpdateBriefAdGroup(briefAdGroup: BriefAdGroup) {
        createOrUpdateBriefAdGroups(listOf(briefAdGroup))
    }

    fun getAdGroups(ids: Collection<Long>): List<Schema.TAdGroup> {
        val rawClients = getObjectsByIds(ids)
        return rawClients.filter { it.protobuf.size() > 0 }.map { transformToBriefAdGroup(it)!! }
    }

    fun getAdGroup(id: Long): Schema.TAdGroup? {
        return getAdGroups(listOf(id)).firstOrNull()
    }

    private fun transformToBriefAdGroup(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TAdGroup? {
        if (raw == null) return null
        return Schema.TAdGroup.parseFrom(raw.protobuf)
    }
}
