package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.client.Schema


data class BriefBanner(
    val id: Long,
    val adGroupId: Long,
    val briefId: Long,
    val source: Banner.EBannerSource,
    val assetIds: List<Long>,
    val assetLinksIds: List<Long>,
    val status: Banner.TBannerSpec.EBannerStatus? = null,
)

/**
 * Баннер со списком ассетов, из которого он был сгенерирован
 * Лежит в таблице banner, настоящий баннер лежит в таблице banner_v2, в будущем планируется их объединить
 */
class BriefBannerGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<BriefBanner>(grutContext, Schema.EObjectType.OT_BANNER, properties) {
    private val setPaths = listOf("/spec")
    override fun buildIdentity(id: Long): ByteString {
        return Schema.TBannerMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TBannerMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: BriefBanner): ByteString {
        return Schema.TBannerMeta.newBuilder().apply {
            id = obj.id
            adGroupId = obj.adGroupId
            campaignId = obj.briefId
            source = obj.source
        }.build().toByteString()
    }

    override fun serializeSpec(obj: BriefBanner): ByteString {
        return Banner.TBannerSpec.newBuilder().apply {
            obj.status?.let { status = it }
            addAllAssetIds(obj.assetIds)
            addAllAssetLinkIds(obj.assetLinksIds)
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TBanner.parseFrom(rawMeta).meta.id
    }

    fun createOrUpdateBriefBanners(briefBanners: List<BriefBanner>) {
        createOrUpdateObjects(briefBanners, setPaths)
    }

    fun createOrUpdateBriefBanner(briefBanner: BriefBanner) {
        createOrUpdateBriefBanners(listOf(briefBanner))
    }

    fun getBanners(ids: Collection<Long>): List<Schema.TBanner> {
        val rawClients = getObjectsByIds(ids)
        return rawClients.filter { it.protobuf.size() > 0 }.map { transformToBriefBanner(it)!! }
    }

    fun selectBanners(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec"),
        index: String? = null,
        limit: Long? = null,
        continuationToken: String? = null,
        allowFullScan: Boolean = false
    ): List<Schema.TBanner> {
        return selectObjectsAs(filter, attributeSelector, index, limit, continuationToken, allowFullScan, ::transformToBriefBanner)
    }

    private fun transformToBriefBanner(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TBanner? {
        if (raw == null) return null
        return Schema.TBanner.parseFrom(raw.protobuf)
    }
}
