package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import com.google.protobuf.InvalidProtocolBufferException
import com.google.protobuf.util.JsonFormat
import org.slf4j.LoggerFactory
import ru.yandex.direct.autobudget.restart.service.Reason
import ru.yandex.direct.core.entity.additionaltargetings.model.CampAdditionalTargeting
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAllowedPageIds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithAttributionModel
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBannerHrefParams
import ru.yandex.direct.core.entity.campaign.model.CampaignWithBrandSafety
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCalltrackingOnSite
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDayBudget
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisabledDomainsAndSsp
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisabledVideoPlacements
import ru.yandex.direct.core.entity.campaign.model.CampaignWithDisallowedPageIds
import ru.yandex.direct.core.entity.campaign.model.CampaignWithEshowsSettings
import ru.yandex.direct.core.entity.campaign.model.CampaignWithFrontpageTypes
import ru.yandex.direct.core.entity.campaign.model.CampaignWithImpressionRate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMetrikaCounters
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalAddMetrikaTagToUrl
import ru.yandex.direct.core.entity.campaign.model.CampaignWithOptionalAddOpenstatTagToUrl
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPlacementTypes
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage
import ru.yandex.direct.core.entity.campaign.model.CampaignWithSiteMonitoring
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithWidgetPartnerId
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.InternalCampaign
import ru.yandex.direct.core.entity.campaign.model.InternalCampaignWithRestriction
import ru.yandex.direct.core.entity.campaign.model.InternalCampaignWithRotationGoalId
import ru.yandex.direct.core.entity.campaign.model.PlacementType
import ru.yandex.direct.core.entity.campaign.model.WalletTypedCampaign
import ru.yandex.direct.core.entity.campaign.model.WithAbSegmentRetargetingConditionIds
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType
import ru.yandex.direct.core.entity.mobileapp.model.SkAdNetworkSlot
import ru.yandex.direct.core.entity.timetarget.model.GeoTimezone
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.MAX_RF_RESET
import ru.yandex.direct.core.grut.api.utils.bigDecimalToGrut
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeToGrut
import ru.yandex.direct.core.grut.api.utils.moscowLocalDateToGrut
import ru.yandex.direct.core.grut.api.utils.toGrutMoney
import ru.yandex.direct.core.grut.api.utils.toIsoCode
import ru.yandex.direct.libs.timetarget.TimeTargetUtils.defaultTimeTarget
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.eshowsVideoTypeToGrut
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.restrictionTypeToGrut
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.rfCloseByClickToGrut
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.toGrutAttributionModel
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.toGrutCampaignMetatype
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.toGrutPlatform
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.toGrutSource
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers.Companion.toGrutStrategyType
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8
import ru.yandex.direct.utils.TimeConvertUtils
import ru.yandex.grut.auxiliary.proto.RfOptions
import ru.yandex.grut.auxiliary.proto.TargetingExpression
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.CampaignPlatform
import ru.yandex.grut.objects.proto.CampaignV2
import ru.yandex.grut.objects.proto.CampaignV2.TCampaignV2Spec
import ru.yandex.grut.objects.proto.CampaignV2.TCampaignV2Spec.TMeaningfulGoal
import ru.yandex.grut.objects.proto.CampaignV2.TCampaignV2Spec.TPlacementTypes
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration
import java.time.LocalDateTime
import java.util.Objects

data class AutoBudgetRestartData(
    val restartTime: LocalDateTime,
    val softRestartTime: LocalDateTime,
    val restartReason: Reason,
    val hasMoney: Boolean,
    val stopTime: LocalDateTime?,
)

data class CampaignGrutModel(
    val campaign: CommonCampaign,
    val orderType: Int?,
    val minusPhraseId: Long? = null,
    val walletId: Long = 0,
    val timeZone: GeoTimezone? = null,
    val autoBudgetRestart: AutoBudgetRestartData? = null,
    val additionalTargetings: List<CampAdditionalTargeting>? = null,
    val skAdNetworkSlot: SkAdNetworkSlot? = null,
    val pricePackageId: Long? = null,
)

class CampaignGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<CampaignGrutModel>(grutContext, Schema.EObjectType.OT_CAMPAIGN_V2, properties) {

    companion object {
        private val logger = LoggerFactory.getLogger(CampaignGrutApi::class.java)
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
        private val IMPRESSION_STANDARD_TYPES = mapOf(
            1000 to CampaignV2.EImpressionStandardType.IST_MRC,
            2000 to CampaignV2.EImpressionStandardType.IST_YANDEX
        )
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TCampaignV2Meta.newBuilder().setId(id).build().toByteString()
    }

    private fun buildDirectIdIdentity(directId: Long): ByteString {
        return Schema.TCampaignV2Meta.newBuilder().setDirectId(directId).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TCampaignV2Meta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: CampaignGrutModel): ByteString {
        val campaign = obj.campaign
        return Schema.TCampaignV2Meta.newBuilder().apply {
            // we don't set orderId because it will be calculated by object_api to preserve constraints
            directId = campaign.id
            directType = CampaignEnumMappers.toGrutCampaignType(campaign.type).number
            source = toGrutSource(campaign.source).number
            clientId = campaign.clientId
            if (Objects.nonNull(campaign.createTime)) {
                creationTime = DateTimeUtils.moscowDateTimeToInstant(campaign.createTime).toEpochMilli()
            }
            campaign.productId?.let { productId = it }
            agencyClientId = campaign.agencyId ?: 0L
            metaType = toGrutCampaignMetatype(campaign.metatype).number
            orderType = obj.orderType ?: 0
        }.build().toByteString()
    }

    fun createOrUpdateCampaigns(campaigns: List<CampaignGrutModel>) {
        createOrUpdateObjects(campaigns, setPaths)
    }

    fun createOrUpdateCampaignsParallel(campaigns: List<CampaignGrutModel>) {
        createOrUpdateObjectsParallel(campaigns, UPDATE_TIMEOUT, setPaths)
    }

    override fun serializeSpec(obj: CampaignGrutModel): ByteString {
        val campaign = obj.campaign
        val timeTarget = getTimeTargetStr(campaign)
        val campaignStrategy = getCampaignStrategy(campaign)
        return TCampaignV2Spec.newBuilder().apply {
            name = campaign.name
            currencyIsoCode = toIsoCode(campaign.currency)
            status = getCampaignStatus(campaign).number
            timeTarget?.let { timeTargetStr = it }
            flags = getCampaignFlags(campaign)
            campaignStrategy?.let { strategy = it }
            (campaign as? CampaignWithCalltrackingOnSite)?.calltrackingSettingsId?.let { calltrackingSettingsId = it }
            (campaign as? CampaignWithAttributionModel)?.attributionModel?.let {
                attribution = toGrutAttributionModel(it).number
            }
            (campaign as? CampaignWithMetrikaCounters)?.metrikaCounters?.let { addAllMetrikaCountersIds(it) }
            campaign.startDate?.let { startDate = moscowLocalDateToGrut(it) }
            campaign.endDate?.let { endDate = moscowLocalDateToGrut(it) }
            obj.minusPhraseId?.let { addMinusPhrasesIds(obj.minusPhraseId) }
            (campaign as? InternalCampaign)?.placeId?.let { placeId = it }
            managerUid = campaign.managerUid ?: 0L
            walletCampaignId = obj.walletId
            getPlatform(campaign)?.let { platform = it.number }
            obj.timeZone?.let {
                timezone = TCampaignV2Spec.TTimezone.newBuilder().apply {
                    countryId = it.regionId
                    name = it.timezone.id
                }.build()
            }
            obj.autoBudgetRestart?.let { autobudgetRestart = getAutobudgetRestartData(it) }
            getPlacementTypes(campaign)?.let { placementTypes = it }
            addAllMeaningfulGoals(getMeaningfulGoals(campaign))
            getMeaningfulGoalsHash(campaign)?.let { meaningfulGoalsHash = it }
            getMoneyParams(campaign)?.let { moneyParams = it }
            getInternalAdOptions(campaign)?.let { internalAdOptions = it }
            getRfOptions(campaign)?.let { rfOptions = it }
            (campaign as? CampaignWithPackageStrategy)?.strategyId?.let { strategyId = it }
            (campaign as? CampaignWithDisabledDomainsAndSsp)?.let { campaignWithDisabledDomainsAndSsp ->
                campaignWithDisabledDomainsAndSsp.disabledDomains?.let { addAllForbiddenDomains(it) }
                campaignWithDisabledDomainsAndSsp.disabledSsp?.let { addAllForbiddenSsp(it) }
            }
            obj.additionalTargetings?.let { additionalTargetingExpression = getAdditionalTargetings(it) }
            getImpressionStandardType(campaign)?.let { impressionStandardType = it.number }
            phraseQualityClustering = getPhraseQualityClustering(campaign).number
            (campaign as? CampaignWithWidgetPartnerId)?.widgetPartnerId?.let { widgetPartnerId = it }
            (campaign as? CampaignWithAllowedPageIds)?.allowedDomains?.let { addAllAllowedDomains(it) }
            (campaign as? CampaignWithAllowedPageIds)?.allowedPageIds?.let { addAllAllowedPageIds(it) }
            (campaign as? CampaignWithEshowsSettings)?.eshowsSettings?.videoType?.let {
                eshowVideoType = eshowsVideoTypeToGrut(it).number
            }
            (campaign as? WithAbSegmentRetargetingConditionIds)?.let {
                it.abSegmentRetargetingConditionId?.let { value -> abSegmentRetargetingConditionId = value }
                it.abSegmentStatisticRetargetingConditionId?.let { value ->
                    abSegmentStatisticsRetargetingConditionId = value
                }
            }
            obj.skAdNetworkSlot?.let {
                skadNetwork = TCampaignV2Spec.TSkAdNetwork.newBuilder().apply {
                    bundleId = it.appBundleId
                    slot = it.slot
                }.build()
            }
            (campaign as? CampaignWithPricePackage)?.auctionPriority?.let { auctionPriority = it.toInt() }
            (campaign as? CampaignWithBannerHrefParams)?.bannerHrefParams?.let { trackingParams = it }
            (campaign as? CampaignWithDisabledVideoPlacements)?.disabledVideoPlacements?.let {
                addAllForbiddenVideoPlacements(it)
            }
            campaign.disabledIps?.let { addAllForbiddenIps(it) }
            getFrontPagePlacements(campaign)?.let { frontpagePlacements = it }
            (campaign as? CampaignWithBrandSafety)?.brandSafetyRetCondId?.let { brandsafetyRetargetingConditionId = it }
            (campaign as? CampaignWithDisallowedPageIds)?.disallowedPageIds?.let { addAllForbiddenPageIds(it) }
            obj.pricePackageId?.let {
                packageId = it
            }
        }.build().toByteString()
    }

    /**
     * В репликации разрешается менять поле source
     * это может происходить, когда коммандер сначала создает камапнию с source = 'direct', а после меняет на 'dc'
     * Кампания может быть создана пустой, а потом на ней могут заполнить agency_client_id, поэтому обновляем так же
     * agency_client_id
     */
    private val setPaths = listOf("/spec", "/meta/source", "/meta/order_type", "/meta/agency_client_id")
    fun createOrUpdateCampaign(obj: CampaignGrutModel) {
        createOrUpdateObject(obj, setPaths)
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TCampaignV2.parseFrom(rawMeta).meta.id
    }

    fun getCampaign(id: Long): Schema.TCampaignV2? {
        return getObjectAs(id, ::transformToCampaign)
    }

    fun getCampaigns(ids: Collection<Long>): List<Schema.TCampaignV2> {
        val rawCampaigns = getObjectsByIds(ids)
        return rawCampaigns.filter { it.protobuf.size() > 0 }.map { transformToCampaign(it)!! }
    }

    fun deleteCampaignsByDirectIds(directIds: List<Long>) {
        val identities = directIds.map { buildDirectIdIdentity(it) }
        return deleteObjectsByIdentities(identities, true)
    }

    fun getCampaignIdsByDirectIds(directIds: Collection<Long>): Map<Long, Long> {
        val identities = directIds.map { buildDirectIdIdentity(it) }
        val rawCampaigns = getObjects(identities, listOf("/meta/id", "/meta/direct_id"), skipNonexistent = true)
        return rawCampaigns
            .map { transformToCampaign(it)!! }
            .associate { it.meta.directId to it.meta.id }
    }

    fun getCampaignsByDirectIds(
        directIds: Collection<Long>,
        attributeSelector: List<String> = listOf("/meta", "/spec")
    ): List<Schema.TCampaignV2> {
        val identities = directIds.map { buildDirectIdIdentity(it) }
        val rawCampaigns = getObjects(identities, attributeSelector)
        return rawCampaigns.filter { it.protobuf.size() > 0 }
            .map { transformToCampaign(it)!! }
    }

    // Для тестов, в которых создается новый маппинг directId -> orderId не по правилу orderId = campaignId + 100_000_000 метод работать не будет
    // Cуществует конечный набор кампаний, у которых orderId вычисляется не по правилу выше, и новых быть не может
    fun getCampaignByDirectId(directId: Long): Schema.TCampaignV2? {
        return getCampaignsByDirectIds(listOf(directId)).firstOrNull()
    }

    private fun transformToCampaign(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TCampaignV2? {
        if (raw == null) return null
        return Schema.TCampaignV2.parseFrom(raw.protobuf)
    }

    private fun getTimeTargetStr(campaign: CommonCampaign): String? {
        if (campaign.timeTarget == null || campaign.timeTarget == defaultTimeTarget()) {
            return null
        }
        // нельзя использовать метод toRawFormat(), так как он ко всем таргетам без пресета добавляет дефолтный пресет
        // а мы хотим иметь строку такую же как в mysql
        return campaign.timeTarget.originalTimeTarget
    }

    fun getCampaignStatus(campaign: CommonCampaign): CampaignV2.ECampaignStatus {
        if (campaign.statusEmpty == true) {
            return CampaignV2.ECampaignStatus.CST_CREATING
        }
        if (campaign.statusArchived == true) {
            return CampaignV2.ECampaignStatus.CST_ARCHIVED
        }
        if (campaign.statusModerate == CampaignStatusModerate.NEW) {
            return CampaignV2.ECampaignStatus.CST_DRAFT
        }
        if (campaign.statusShow == false) {
            return CampaignV2.ECampaignStatus.CST_STOPPED
        }
        return CampaignV2.ECampaignStatus.CST_ACTIVE
    }

    private fun getPlatform(campaign: CommonCampaign): CampaignPlatform.ECampaignPlatform? {
        if (campaign is CampaignWithStrategy) {
            return campaign.strategy.platform?.let { toGrutPlatform(it) }
        }
        return null
    }

    private fun getCampaignFlags(campaign: CommonCampaign): TCampaignV2Spec.TFlags {
        return TCampaignV2Spec.TFlags.newBuilder().apply {
            isVirtual = campaign.isVirtual ?: false
            isAloneTrafaretAllowed = campaign.isAloneTrafaretAllowed ?: false
            requireFiltrationByDontShowDomains = campaign.requireFiltrationByDontShowDomains ?: false
            noTitleSubstitute = !(campaign.hasTitleSubstitution ?: true)
            hidePermalinkInfo = !(campaign.enableCompanyInfo ?: true)
            noExtendedGeotargeting = !(campaign.hasExtendedGeoTargeting ?: true)
            enableCpcHold = campaign.enableCpcHold ?: false
            isAllowedOnAdultContent = campaign.isAllowedOnAdultContent ?: false
            addMetrikaTagToUrl = (campaign as? CampaignWithOptionalAddMetrikaTagToUrl)?.hasAddMetrikaTagToUrl == true
            siteMonitoringEnabled = (campaign as? CampaignWithSiteMonitoring)?.hasSiteMonitoring == true
            openStatEnabled = (campaign as? CampaignWithOptionalAddOpenstatTagToUrl)?.hasAddOpenstatTagToUrl == true
            isNewIosVersionEnabled = campaign.isNewIosVersionEnabled == true
            hasTurboSmarts = campaign.hasTurboSmarts == true
            hasTurboApp = campaign.hasTurboApp == true
            isSkadNetworkEnabled = campaign.isSkadNetworkEnabled == true
            isWorldwide = campaign.isWwManagedOrder == true
        }.build()
    }

    private fun getMoneyParams(campaign: CommonCampaign): TCampaignV2Spec.TMoneyParams? {
        val isSumAggregated: Boolean? = if (campaign is WalletTypedCampaign) campaign.isSumAggregated == true else null
        return TCampaignV2Spec.TMoneyParams.newBuilder().apply {
            sum = toGrutMoney(campaign.sum)
            sumSpent = toGrutMoney(campaign.sumSpent)
            (campaign as? CampaignWithDayBudget)?.dayBudget?.let { dayBudget = toGrutMoney(it) }
            flags = TCampaignV2Spec.TMoneyParams.TMoneyProcessingFlags.newBuilder().apply {
                isSumAggregated?.let { isWalletSumAggregated = it }
                campaign.paidByCertificate?.let { isPaidByCertificate = it }
            }.build()
        }.build()
    }

    private fun getCampaignStrategy(campaign: CommonCampaign): TCampaignV2Spec.TStrategy? {
        if (campaign !is CampaignWithStrategy) {
            return null
        }
        val strategyData = campaign.strategy.strategyData
        return TCampaignV2Spec.TStrategy.newBuilder().apply {
            campaign.strategy.strategyName?.let { type = toGrutStrategyType(it).number }
            strategyData.autoProlongation?.let { autoProlongation = it == 1L }
            strategyData.avgBid?.let { avgBid = toGrutMoney(it) }
            strategyData.avgCpa?.let { avgCpa = toGrutMoney(it) }
            strategyData.avgCpi?.let { avgCpi = toGrutMoney(it) }
            strategyData.avgCpm?.let { avgCpm = toGrutMoney(it) }
            strategyData.avgCpv?.let { avgCpv = toGrutMoney(it) }
            strategyData.bid?.let { bid = toGrutMoney(it) }
            strategyData.budget?.let { budget = toGrutMoney(it) }
            strategyData.crr?.let { crr = it.toInt() }
            strategyData.dailyChangeCount?.let { dailyChangeCount = it.toInt() }
            strategyData.filterAvgBid?.let { filterAvgBid = toGrutMoney(it) }
            strategyData.filterAvgCpa?.let { filterAvgCpa = toGrutMoney(it) }
            strategyData.start?.let { startDate = moscowLocalDateToGrut(it) }
            strategyData.finish?.let { finishDate = moscowLocalDateToGrut(it) }
            strategyData.goalId?.let { goalId = it }
            strategyData.lastBidderRestartTime?.let { bidderRestartTime = moscowDateTimeToGrut(it) }
            strategyData.lastUpdateTime?.let { updateTime = moscowDateTimeToGrut(it) }
            strategyData.name?.let { name = it }
            strategyData.payForConversion?.let { payForConversion = it }
            strategyData.profitability?.let { profitability = bigDecimalToGrut(it) }
            strategyData.reserveReturn?.let { reserveReturn = it.toInt() }
            strategyData.roiCoef?.let { roiCoef = bigDecimalToGrut(it) }
            strategyData.sum?.let { budgetLimit = toGrutMoney(it) }
            campaign.strategy.autobudget?.let { autobudgetEnabled = it == CampaignsAutobudget.YES }
        }.build()
    }

    private fun getInternalAdOptions(campaign: CommonCampaign): TCampaignV2Spec.TInternalAdOptions? {
        if (campaign !is InternalCampaign) {
            return null
        }
        return TCampaignV2Spec.TInternalAdOptions.newBuilder().apply {
            if (campaign is InternalCampaignWithRestriction) {
                campaign.restrictionType?.let { restrictionType = restrictionTypeToGrut(it).number }
                campaign.restrictionValue?.let { restrictionValue = it }
            }
            campaign.rfCloseByClick?.let { rfCloseByClick = rfCloseByClickToGrut(it).number }
            if (campaign is InternalCampaignWithRotationGoalId) {
                campaign.rotationGoalId?.let { rotationGoalId = it }
            }
            isMobile = campaign.isMobile
        }.build()
    }

    private fun getRfOptions(campaign: CommonCampaign): RfOptions.TRfOptions? {
        if (campaign !is CampaignWithImpressionRate) {
            return null
        }
        if (campaign.impressionRateCount ?: 0 == 0) {
            return null
        }

        // TODO объединить код с тем, что в группах
        // на данный момент есть отличия
        return RfOptions.TRfOptions.newBuilder().apply {
            shows = RfOptions.TRfOptions.TLimit.newBuilder().apply {
                count = campaign.impressionRateCount
                val periodInDays =
                    if (campaign.impressionRateIntervalDays ?: 0 == 0) MAX_RF_RESET else campaign.impressionRateIntervalDays
                period = TimeConvertUtils.daysToSecond(periodInDays).toInt()
            }.build()
        }.build()

        // TODO добавить логику для новых rf внутренней рекламы из campaigns_internal
    }

    private fun getAutobudgetRestartData(autoBudgetRestart: AutoBudgetRestartData) =
        TCampaignV2Spec.TAutoBudgetRestart.newBuilder().apply {
            restartTime = moscowDateTimeToGrut(autoBudgetRestart.restartTime)
            softRestartTime = moscowDateTimeToGrut(autoBudgetRestart.softRestartTime)
            restartReason = CampaignEnumMappers.autobudgetRestartReasonToGrut(autoBudgetRestart.restartReason).number
            hasMoney = autoBudgetRestart.hasMoney
            autoBudgetRestart.stopTime?.let { time -> stopTime = moscowDateTimeToGrut(time) }
        }.build()

    fun getPlacementTypes(campaign: CommonCampaign): TPlacementTypes? {
        if (campaign !is CampaignWithPlacementTypes || campaign.placementTypes.isNullOrEmpty()) {
            return null
        }
        return TPlacementTypes.newBuilder()
            .apply {
                searchPage = campaign.placementTypes.contains(PlacementType.SEARCH_PAGE)
                advGallery = campaign.placementTypes.contains(PlacementType.ADV_GALLERY)
            }.build()
    }

    fun getMeaningfulGoalsHash(campaign: CommonCampaign): Long? {
        return if (campaign is CampaignWithMeaningfulGoals && campaign.rawMeaningfulGoals != null) {
            getMd5HalfHashUtf8(campaign.rawMeaningfulGoals).toLong()
        } else {
            return null
        }
    }

    fun getMeaningfulGoals(campaign: CommonCampaign): List<TMeaningfulGoal> {
        if (campaign !is CampaignWithMeaningfulGoals || campaign.meaningfulGoals.isNullOrEmpty()) {
            return listOf()
        }
        return campaign.meaningfulGoals
            .map {
                TMeaningfulGoal.newBuilder()
                    .apply {
                        goalId = it.goalId
                        conversionValue = toGrutMoney(it.conversionValue)
                        isMetrikaSourceOfValue = it.isMetrikaSourceOfValue == true
                    }.build()
            }
    }

    fun getAdditionalTargetings(additionalTargetings: List<CampAdditionalTargeting>): TargetingExpression.TTargetingExpression {
        val builder = TargetingExpression.TTargetingExpression.newBuilder()
        additionalTargetings.forEach {
            try {
                val partBuilder = TargetingExpression.TTargetingExpression.newBuilder()
                JsonFormat.parser().merge(it.data, partBuilder)
                builder.addAllAnd(partBuilder.build().andList)
            } catch (e: InvalidProtocolBufferException) {
                logger.error("Failed to parse campaign additional targeting ${it.id}")
            }
        }
        return builder.build()
    }

    private fun getImpressionStandardType(campaign: CommonCampaign): CampaignV2.EImpressionStandardType? {
        if (campaign !is CampaignWithImpressionRate) {
            return null
        }
        val impressionRateIntervalDays = campaign.impressionRateIntervalDays ?: 1_000
        return IMPRESSION_STANDARD_TYPES[impressionRateIntervalDays] ?: CampaignV2.EImpressionStandardType.IST_UNKNOWN
    }

    private fun getPhraseQualityClustering(campaign: CommonCampaign): CampaignV2.EPhraseQualityClustering {
        return if (campaign.isOrderPhraseLengthPrecedenceEnabled == true) CampaignV2.EPhraseQualityClustering.PQC_BY_ORDER
        else CampaignV2.EPhraseQualityClustering.PQC_UNKNOWN
    }

    fun getFrontPagePlacements(campaign: CommonCampaign): TCampaignV2Spec.TFrontPagePlacements? {
        if (campaign !is CampaignWithFrontpageTypes) {
            return null
        }
        return TCampaignV2Spec.TFrontPagePlacements.newBuilder().apply {
            frontpage = campaign.allowedFrontpageType.contains(FrontpageCampaignShowType.FRONTPAGE)
            frontpageMobile = campaign.allowedFrontpageType.contains(FrontpageCampaignShowType.FRONTPAGE_MOBILE)
            browserNewTab = campaign.allowedFrontpageType.contains(FrontpageCampaignShowType.BROWSER_NEW_TAB)
        }.build()
    }

    fun updateCampaigns(objects: Collection<UpdatedObject>) =
        updateObjects(objects)
}
