package ru.yandex.direct.core.grut.api

import com.google.protobuf.Timestamp
import java.time.Instant
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toUacYdbCampaignHistory
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbUtils
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaign
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService
import ru.yandex.direct.core.grut.api.GrutApiBase.Companion.GRUT_GET_OBJECTS_ATTEMPTS
import ru.yandex.direct.core.grut.model.GrutHistoryEventEntry
import ru.yandex.direct.core.grut.model.GrutHistoryEventType
import ru.yandex.direct.validation.defect.CommonDefects
import ru.yandex.direct.validation.defect.DateDefects.broadHistoryTimeInterval
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.grut.client.GrutClient
import ru.yandex.grut.history.History
import ru.yandex.grut.objects.proto.client.Schema

class CampaignGrutHistoryApi(
    grutClient: GrutClient,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutUacCampaignService: GrutUacCampaignService,
) : GrutHistoryBaseApi(grutClient, Schema.EObjectType.OT_CAMPAIGN) {

    companion object {
        const val ITERATION_LIMIT = 3
        const val LIMIT = 2000
    }

    /**
     * Метод возвращает список изменений данной кампании в порядке возрастания времени фиксации события
     * @param campaignId - идентификатор кампании
     * @param fromTime - нижняя граница времени фиксации события (может быть не определена)
     * @param toTime - верхняя граница времени фиксации события
     * @return список событий, зафиксированных в [fromTime, toTime]
     */
    fun getCampaignHistory(
        campaignId: Long,
        fromTime: Long?,
        toTime: Long
    ): ValidationResult<List<GrutHistoryEventEntry<UacYdbCampaign>>, Defect<*>> {

        val identity = Schema.TCampaignMeta.newBuilder()
            .setId(campaignId)
            .build()
            .toByteString()

        val campaign = grutUacCampaignService.getCampaignById(campaignId.toString())
            ?: return ValidationResult.failed(null, CommonDefects.objectNotFound())

        val interval = History.TTimeInterval.newBuilder().apply {
            begin = Timestamp.newBuilder().apply {
                seconds = UacYdbUtils.toEpochSecond(campaign.createdAt.minusSeconds(1))
            }.build()
            end = Timestamp.newBuilder().apply {
                seconds = toTime
            }.build()
        }.build()

        var currentIteration = 0
        var continuationToken: String? = null
        val events = mutableListOf<GrutHistoryEventEntry<UacYdbCampaign>>()

        val minTime = fromTime ?: UacYdbUtils.toEpochSecond(campaign.createdAt)
        while (true) {
            val result = grutTransactionProvider.runRetryable(GRUT_GET_OBJECTS_ATTEMPTS) {
                selectObjectsHistory(identity, interval, continuationToken, LIMIT)
            }

            continuationToken = result.continuationToken

            result.events
                .filterNot { it.protobuf == null }
                .forEach {
                    val campaignHistoryEvent = toGrutCampaignHistory(campaign, it)
                    events.add(campaignHistoryEvent)
                    if (campaignHistoryEvent.timestamp < minTime) {
                        return ValidationResult.success(events.reversed())
                    }
                }

            if (result.events.size < LIMIT) return ValidationResult.success(events.reversed())

            if (currentIteration++ >= ITERATION_LIMIT) {
                return ValidationResult.failed(null, broadHistoryTimeInterval(
                    fromTime?.let { UacYdbUtils.fromEpochSecond(fromTime) },
                    UacYdbUtils.fromEpochSecond(toTime)))
            }
        }
    }

    private fun toGrutCampaignHistory(
        campaign: UacYdbCampaign,
        raw: History.THistoryEvent
    ): GrutHistoryEventEntry<UacYdbCampaign> {
        val campaignHistory = Schema.TCampaign.parseFrom(raw.protobuf).toUacYdbCampaignHistory(campaign)
        return GrutHistoryEventEntry(
            campaignHistory,
            toEpochSeconds(raw.time),
            GrutHistoryEventType.valueOf(raw.eventType.name),
        )
    }

    private fun toEpochSeconds(timestamp: Timestamp): Long {
        val currentInstant = Instant.ofEpochSecond(
            timestamp.seconds,
            timestamp.nanos.toLong())
        return currentInstant.epochSecond
    }
}
