package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import com.google.protobuf.InvalidProtocolBufferException
import com.google.protobuf.util.JsonFormat
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.additionaltargetings.model.ClientAdditionalTargeting
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.model.ClientNds
import ru.yandex.direct.core.entity.client.model.TinType
import ru.yandex.direct.core.entity.product.model.ProductType
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.bigDecimalToGrut
import ru.yandex.direct.core.grut.api.utils.moscowLocalDateToGrut
import ru.yandex.direct.core.grut.api.utils.toGrutMoney
import ru.yandex.direct.mysql2grut.enummappers.CampaignEnumMappers
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.grut.auxiliary.proto.TargetingExpression
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TVersionedPayload
import ru.yandex.grut.objects.proto.Client.TBalanceOptions
import ru.yandex.grut.objects.proto.Client.TBillingAggregate
import ru.yandex.grut.objects.proto.Client.TClientFlags
import ru.yandex.grut.objects.proto.Client.TClientNds
import ru.yandex.grut.objects.proto.Client.TClientSpec
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration

data class ClientGrutModel(
    val client: Client,
    val ndsHistory: List<ClientNds>,
    val billingAggregates: List<BillingAggregateInfo> = emptyList(),
    val internalAdProduct: String? = null,
    val additionalTargetings: List<ClientAdditionalTargeting>? = null
)

data class BillingAggregateInfo(
    val id: Long,
    val walletOrderId: Long,
    val productId: Long,
    val productType: ProductType,
)

class ClientGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<ClientGrutModel>(grutContext, Schema.EObjectType.OT_CLIENT, properties) {

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
        private val logger = LoggerFactory.getLogger(ClientGrutApi::class.java)
        private val clientSpecFields = listOf(
            "name",
            "chief_uid",
            "flags",
            "client_nds_history",
            "billing_aggregates",
            "internal_ad_distribution_tag",
            "auto_overdraft_limit",
            "balance_options",
            "tin",
            "tin_type",
            "additional_targeting_expression",
            "default_allowed_domains"
        )
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TClientMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TClientMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: ClientGrutModel): ByteString {
        return Schema.TClientMeta.newBuilder()
            .apply {
                id = obj.client.id
                //object_api принимает дату в миллисекундах, а отдает в микросекундах  YTORM-275
                obj.client.createDate?.let { creationTime = DateTimeUtils.moscowDateTimeToInstant(it).toEpochMilli() }
            }
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: ClientGrutModel): ByteString {
        return serializeSpecProto(obj).toByteString()
    }

    private fun serializeSpecProto(obj: ClientGrutModel): TClientSpec {
        val client = obj.client
        val ndsHistory = getClientNdsHistory(obj.ndsHistory)
        val billingAggregates = getBillingAggregates(obj.billingAggregates)
        return TClientSpec.newBuilder().apply {
            if (client.name != null) {
                name = client.name
            }
            if (client.chiefUid != null) {
                chiefUid = client.chiefUid
            }
            flags = getClientFlags(client)
            client.autoOverdraftLimit?.let { autoOverdraftLimit = toGrutMoney(it) }
            balanceOptions = getBalanceOptions(client)
            addAllClientNdsHistory(ndsHistory)
            addAllBillingAggregates(billingAggregates)
            if (obj.internalAdProduct != null) {
                internalAdDistributionTag = obj.internalAdProduct
            }
            client.tin?.let { tin = client.tin }
            if (client.tinType == TinType.LEGAL) {
                tinType = TClientSpec.ETinType.TT_LEGAL.number
            } else if (client.tinType == TinType.PHYSICAL) {
                tinType = TClientSpec.ETinType.TT_PHYSICAL.number
            }
            obj.additionalTargetings?.let { additionalTargetingExpression = getAdditionalTargetings(it) }
            client.defaultAllowedDomains?.let { addAllDefaultAllowedDomains(it) }
        }
            .build()
    }

    fun createOrUpdateClients(clients: List<ClientGrutModel>) {
        val payloads = clients.map { serializeClient(it) }
        createOrUpdateObjects(payloads)
    }

    fun createOrUpdateClient(client: ClientGrutModel) {
        createOrUpdateClients(listOf(client))
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TClient.parseFrom(rawMeta).meta.id
    }

    fun getClient(id: Long): Schema.TClient? {
        return getObjectAs(id, ::transformToClient)
    }

    fun getClients(ids: List<Long>): List<Schema.TClient> {
        val rawClients = getObjectsByIds(ids)
        return rawClients.filter { it.protobuf.size() > 0 }.map { transformToClient(it)!! }
    }

    fun selectClients(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec"),
        index: String? = null,
        limit: Long? = null,
        continuationToken: String? = null,
        allowFullScan: Boolean = false
    ): List<Schema.TClient> {
        return selectObjectsAs(
            filter,
            attributeSelector,
            index,
            limit,
            continuationToken,
            allowFullScan,
            ::transformToClient
        )
    }

    fun createOrUpdateClientsParallel(clients: List<ClientGrutModel>) {
        val payloads = clients.map { serializeClient(it) }
        createOrUpdateObjectsParallel(payloads, UPDATE_TIMEOUT)
    }

    private fun serializeClient(client: ClientGrutModel): UpdatedObject {
        val meta = serializeMeta(client)
        val spec = serializeSpecProto(client)

        val setRemove = prepareSetRemovePaths(spec, "/spec", clientSpecFields)
        setRemove.pathsToSet.add("/meta/creation_time")

        return UpdatedObject(
            meta = meta,
            spec = spec.toByteString(),
            setPaths = setRemove.pathsToSet,
            removePaths = setRemove.pathsToRemove
        )
    }

    private fun transformToClient(raw: TVersionedPayload?): Schema.TClient? {
        if (raw == null) return null
        return Schema.TClient.parseFrom(raw.protobuf)
    }

    private fun getClientFlags(client: Client) =
        TClientFlags.newBuilder().apply {
            nonResident = client.nonResident == true
            hideMarketRating = client.hideMarketRating == true
            isFaviconBlocked = client.faviconBlocked == true
            asSoonAsPossible = client.asSoonAsPossible == true
            businessUnit = client.isBusinessUnit == true
            socialAdvertising = client.socialAdvertising == true
        }.build()

    private fun getClientNdsHistory(ndsHistory: List<ClientNds>) =
        ndsHistory.sortedBy { it.dateFrom }.map {
            TClientNds.newBuilder().apply {
                nds = bigDecimalToGrut(it.nds.asPercent())
                dateFrom = moscowLocalDateToGrut(it.dateFrom)
                dateTo = moscowLocalDateToGrut(it.dateTo)
            }.build()
        }

    private fun getBillingAggregates(billingAggregates: List<BillingAggregateInfo>) =
        billingAggregates
            .sortedBy { it.id }
            .map {
                TBillingAggregate.newBuilder()
                    .apply {
                        billingAggregateCampaignId = it.id
                        walletCampaignId = it.walletOrderId
                        productId = it.productId
                        productType = CampaignEnumMappers.toGrutProductType(it.productType).number
                    }
                    .build()
            }

    private fun getBalanceOptions(client: Client) =
        TBalanceOptions.newBuilder().apply {
            client.overdraftLimit?.let { overdraftLimit = toGrutMoney(it) }
            client.debt?.let { debt = toGrutMoney(it) }
            isBannedInBalance = client.statusBalanceBanned == true
        }.build()

    private fun getAdditionalTargetings(additionalTargetings: List<ClientAdditionalTargeting>): TargetingExpression.TTargetingExpression {
        val builder = TargetingExpression.TTargetingExpression.newBuilder()
        additionalTargetings.forEach {
            try {
                val partBuilder = TargetingExpression.TTargetingExpression.newBuilder()
                JsonFormat.parser().merge(it.data, partBuilder)
                builder.addAllAnd(partBuilder.build().andList)
            } catch (e: InvalidProtocolBufferException) {
                logger.error("Failed to parse client additional targeting ${it.id}")
            }
        }
        return builder.build()
    }
}
