package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.conversionsource.model.ConversionAction
import ru.yandex.direct.core.entity.conversionsource.model.ConversionActionValue
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSource
import ru.yandex.direct.core.entity.conversionsource.model.ConversionSourceSettings
import ru.yandex.direct.core.entity.conversionsource.model.Destination
import ru.yandex.direct.core.entity.conversionsource.model.ProcessingInfo
import ru.yandex.direct.core.entity.conversionsource.model.ProcessingStatus
import ru.yandex.direct.core.entity.conversionsourcetype.model.ConversionSourceTypeCode
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.moneyFromGrut
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeFromGrut
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeToGrut
import ru.yandex.direct.core.grut.api.utils.toCurrencyCode
import ru.yandex.direct.core.grut.api.utils.toGrutMoney
import ru.yandex.direct.core.grut.api.utils.toIsoCode
import ru.yandex.direct.currency.Money
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceMetaBase.ETypeCode
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.DestinationCase
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.SettingsCase
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.TConversionAction
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.TPostBackDestination
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.TProcessingInfo
import ru.yandex.grut.objects.proto.ConversionSource.TConversionSourceSpec.TProcessingInfo.EProcessingStatus
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.client.Schema.TConversionSourceMeta

private val LOGGER = LoggerFactory.getLogger(ConversionSourceGrutApi::class.java)

class ConversionSourceGrutApi(
    val grutContext: GrutContext,
    properties: GrutApiProperties = DefaultGrutApiProperties(),
) :
    GrutApiBase<ConversionSource>(grutContext, Schema.EObjectType.OT_CONVERSION_SOURCE, properties) {
    companion object {
        const val CONVERSION_SOURCE_FETCH_LIMIT = 2000L
        val specSetPathOnUpdate = listOf(
            "/spec/name",
            "/spec/conversion_actions",
            "/spec/update_period_hours",
        )
        val resetOnUpdate = listOf(
            "/spec/processing_info/last_scheduled_time",
            "/spec/processing_info/next_scheduled_time",
        )
    }

    override fun buildIdentity(id: Long): ByteString =
        TConversionSourceMeta.newBuilder().setId(id).build().toByteString()

    override fun parseIdentity(identity: ByteString): Long = TConversionSourceMeta.parseFrom(identity).id

    override fun serializeMeta(obj: ConversionSource): ByteString = obj.toMetaProto().toByteString()

    override fun serializeSpec(obj: ConversionSource): ByteString = obj.toSpecProto().toByteString()

    override fun getMetaId(rawMeta: ByteString): Long = Schema.TConversionSource.parseFrom(rawMeta).meta.id

    fun selectByClientId(clientId: ClientId, limit: Long = CONVERSION_SOURCE_FETCH_LIMIT): List<ConversionSource> {
        return selectObjectsAs("[/meta/client_id] = $clientId", limit = limit, transform = ::transformToModel)
    }

    fun selectByIds(ids: Collection<Long>): List<ConversionSource> {
        if (ids.isEmpty()) {
            return listOf()
        }
        return getObjectsByIds(ids, attributeSelector = listOf("/meta", "/spec", "/status"), skipNonexistent = true)
            .mapNotNull { transformToModel(it) }
    }

    fun selectByFilter(
        filter: String,
        limit: Long = CONVERSION_SOURCE_FETCH_LIMIT,
        continuationToken: String? = null,
    ): Pair<String, List<ConversionSource>> {
        val response =
            grutContext.client.selectObjects(ObjectApiServiceOuterClass.TReqSelectObjects.newBuilder().apply {
                objectType = Schema.EObjectType.OT_CONVERSION_SOURCE
                this.filter = filter
                this.limit = limit
                continuationToken?.let { this.continuationToken = continuationToken }
                addAllAttributeSelector(listOf("/meta", "/spec", "/status"))
                this.allowFullScan = true
            }.build())
        return response.continuationToken to response.payloadsList.mapNotNull { transformToModel(it) }
    }

    fun updateConversionSources(conversionSources: Collection<ConversionSource>) {
        val updatedObjects: Collection<UpdatedObject> = conversionSources.map {
            UpdatedObject(
                meta = serializeMeta(it),
                spec = serializeSpec(it),
                setPaths = expandUpdatePathsWithOneOf(specSetPathOnUpdate, it),
                removePaths = resetOnUpdate,
            )
        }
        updateObjects(updatedObjects)
    }

    fun updateConversionActions(conversionSources: Collection<ConversionSource>) {
        updateObjects(conversionSources, listOf("/spec/conversion_actions"), resetOnUpdate)
    }

    fun delete(ids: List<Long>) {
        val identities = ids.map { buildIdentity(it) }
        return deleteObjectsByIdentities(identities, true)
    }

    private fun transformToModel(raw: ObjectApiServiceOuterClass.TVersionedPayload?): ConversionSource? {
        if (raw == null) return null
        val parsedOffer = Schema.TConversionSource.parseFrom(raw.protobuf) ?: return null
        return parsedOffer.toCore()
    }

    private fun expandUpdatePathsWithOneOf(
        baseUpdatePaths: List<String>,
        conversionSource: ConversionSource,
    ): Collection<String> {
        val updatePaths: MutableList<String> = baseUpdatePaths.map { it }.toMutableList()
        when (conversionSource.settings) {
            is ConversionSourceSettings.Ftp -> updatePaths.add("/spec/ftp")
            is ConversionSourceSettings.GoogleSheets -> updatePaths.add("/spec/google_sheets")
            is ConversionSourceSettings.Link -> updatePaths.add("/spec/link")
            is ConversionSourceSettings.Metrika -> updatePaths.add("/spec/metrika")
            is ConversionSourceSettings.SFtp -> updatePaths.add("/spec/sftp")
        }
        when (conversionSource.destination) {
            is Destination.CrmApi -> updatePaths.add("/spec/crm_api_destination")
            is Destination.PostBack -> updatePaths.add("/spec/post_back_destination")
        }
        return updatePaths
    }
}

private fun ConversionSource.toMetaProto(): TConversionSourceMeta {
    val obj = this
    return TConversionSourceMeta.newBuilder().apply {
        if (obj.id != null && obj.id > 0) {
            id = obj.id
        }
        clientId = obj.clientId.asLong()
        typeCode = obj.typeCode.toProto().number
    }.build()
}

private fun ConversionSource.toSpecProto(): TConversionSourceSpec {
    val obj = this
    return TConversionSourceSpec.newBuilder().apply {
        name = obj.name
        when (obj.settings) {
            is ConversionSourceSettings.Link -> link = obj.settings.toProto()
            is ConversionSourceSettings.Metrika -> metrika = obj.settings.toProto()
            is ConversionSourceSettings.GoogleSheets -> googleSheets = obj.settings.toProto()
            is ConversionSourceSettings.Ftp -> ftp = obj.settings.toProto()
            is ConversionSourceSettings.SFtp -> sftp = obj.settings.toProto()
        }
        addAllConversionActions(obj.actions.map { it.toProto() })
        processingInfo = obj.processingInfo.toProto()
        updatePeriodHours = obj.updatePeriodHours
        when (obj.destination) {
            is Destination.CrmApi -> crmApiDestination = obj.destination.toProto()
            is Destination.PostBack -> postBackDestination = obj.destination.toProto()
        }
    }.build()
}

private fun Schema.TConversionSource.toCore(): ConversionSource? {
    val typeCode = ETypeCode.forNumber(meta.typeCode).toCore()
    val destination = extractDestination(spec.destinationCase, spec)
    val settings = extractSettings(spec.settingsCase, spec)
    if (typeCode == null || destination == null || settings == null) {
        return null
    }

    return ConversionSource(
        meta.id,
        typeCode,
        ClientId.fromLong(meta.clientId),
        spec.name,
        settings,
        extractDestinationCounterIdOrZero(spec.destinationCase, spec),
        spec.conversionActionsList.map { it.toCore() },
        spec.updatePeriodHours,
        destination,
        spec.processingInfo.toCore()
    )
}

private fun TConversionAction.toCore(): ConversionAction {
    val goalId = if (hasGoalId()) this.goalId else null
    return ConversionAction(name, goalId, extractConversionActionValue(valueCase, this))
}

fun extractConversionActionValue(
    valueCase: TConversionAction.ValueCase, action: TConversionAction,
): ConversionActionValue? {
    return when (valueCase) {
        TConversionAction.ValueCase.FIXED -> action.fixed.let {
            ConversionActionValue.Fixed(
                Money.valueOf(moneyFromGrut(it.cost), toCurrencyCode(it.currencyIsoCode))
            )
        }
        TConversionAction.ValueCase.DYNAMIC -> action.dynamic.let {
            if (it.hasCurrencyIsoCode() && (it.hasCostFrom() || it.hasCostTo())) {
                val currencyCode = toCurrencyCode(it.currencyIsoCode)
                ConversionActionValue.Dynamic(
                    costFrom = if (it.hasCostFrom()) {
                        Money.valueOf(moneyFromGrut(it.costFrom), currencyCode)
                    } else null,
                    costTo = if (it.hasCostTo()) {
                        Money.valueOf(moneyFromGrut(it.costTo), currencyCode)
                    } else null,
                )
            } else {
                null
            }
        }
        TConversionAction.ValueCase.VALUE_NOT_SET -> null
    }
}

private fun ConversionAction.toProto(): TConversionAction {
    val obj = this
    return TConversionAction.newBuilder().apply {
        name = obj.name
        if (obj.goalId != null) {
            goalId = obj.goalId
        }
        when (obj.value) {
            is ConversionActionValue.Fixed -> fixed = obj.value.toProto()
            is ConversionActionValue.Dynamic -> dynamic = obj.value.toProto()
            null -> {}
            else -> {
                throw IllegalStateException("unsupported conversion action value")
            }
        }
    }.build()
}

private fun ConversionActionValue.Fixed.toProto(): TConversionAction.TValueFixed {
    val obj = this
    return TConversionAction.TValueFixed.newBuilder().apply {
        currencyIsoCode = toIsoCode(obj.cost.currencyCode)
        cost = toGrutMoney(obj.cost.bigDecimalValue())
    }.build()
}

private fun ConversionActionValue.Dynamic.toProto(): TConversionAction.TValueDynamic {
    val obj = this
    val currencyCode = (costFrom ?: costTo)?.currencyCode

    return TConversionAction.TValueDynamic.newBuilder().apply {
        if (currencyCode != null) {
            currencyIsoCode = toIsoCode(currencyCode)
        }
        if (obj.costFrom != null) {
            costFrom = toGrutMoney(obj.costFrom.bigDecimalValue())
        }
        if (obj.costTo != null) {
            costTo = toGrutMoney(obj.costTo.bigDecimalValue())
        }
    }.build()
}

private fun ETypeCode.toCore(): ConversionSourceTypeCode? = when (this) {
    ETypeCode.TC_FTP -> ConversionSourceTypeCode.FTP
    ETypeCode.TC_LINK -> ConversionSourceTypeCode.LINK
    ETypeCode.TC_GOOGLE_SHEETS -> ConversionSourceTypeCode.GOOGLE_SHEETS
    ETypeCode.TC_METRIKA -> ConversionSourceTypeCode.METRIKA
    ETypeCode.TC_SFTP -> ConversionSourceTypeCode.SFTP
    ETypeCode.TC_OTHER -> ConversionSourceTypeCode.API
    else -> {
        LOGGER.warn("unsupported conversion source type code $this")
        null
    }
}

private fun ConversionSourceTypeCode.toProto(): ETypeCode = when (this) {
    ConversionSourceTypeCode.FTP -> ETypeCode.TC_FTP
    ConversionSourceTypeCode.LINK -> ETypeCode.TC_LINK
    ConversionSourceTypeCode.GOOGLE_SHEETS -> ETypeCode.TC_GOOGLE_SHEETS
    ConversionSourceTypeCode.METRIKA -> ETypeCode.TC_METRIKA
    ConversionSourceTypeCode.SFTP -> ETypeCode.TC_SFTP
    ConversionSourceTypeCode.OTHER -> ETypeCode.TC_OTHER
    else -> {
        throw IllegalStateException("unsupported conversion source type code $this")
    }
}

private fun extractSettings(settingsCase: SettingsCase, spec: TConversionSourceSpec): ConversionSourceSettings? {
    return when (settingsCase) {
        SettingsCase.LINK -> ConversionSourceSettings.Link(
            spec.link.url
        )
        SettingsCase.METRIKA -> spec.metrika.let {
            ConversionSourceSettings.Metrika(it.counter, it.domain)
        }
        SettingsCase.GOOGLE_SHEETS -> ConversionSourceSettings.GoogleSheets(
            spec.googleSheets.url
        )
        SettingsCase.FTP -> spec.ftp.let {
            ConversionSourceSettings.Ftp(
                it.host, it.port, it.tls, it.login, it.encryptedPassword, it.path
            )
        }
        SettingsCase.SFTP -> spec.sftp.let {
            ConversionSourceSettings.SFtp(
                it.host, it.port, it.login, it.encryptedPassword, it.path
            )
        }
        else -> {
            LOGGER.warn("unsupported setting in $spec")
            null
        }
    }
}

private fun ConversionSourceSettings.Link.toProto(): TConversionSourceSpec.TLinkSettings {
    val obj = this
    return TConversionSourceSpec.TLinkSettings.newBuilder().apply {
        url = obj.url
    }.build()
}

private fun ConversionSourceSettings.Metrika.toProto(): TConversionSourceSpec.TMetrikaSettings {
    val obj = this
    return TConversionSourceSpec.TMetrikaSettings.newBuilder().apply {
        counter = obj.counterId
        domain = obj.domain
    }.build()
}

private fun ConversionSourceSettings.GoogleSheets.toProto(): TConversionSourceSpec.TGoogleSheetsSettings {
    val obj = this
    return TConversionSourceSpec.TGoogleSheetsSettings.newBuilder().apply {
        url = obj.url
    }.build()
}

private fun ConversionSourceSettings.Ftp.toProto(): TConversionSourceSpec.TFtpSettings {
    val obj = this
    return TConversionSourceSpec.TFtpSettings.newBuilder().apply {
        host = obj.host
        port = obj.port
        tls = obj.tls
        login = obj.login
        encryptedPassword = obj.encryptedPassword
        path = obj.path
    }.build()
}

private fun ConversionSourceSettings.SFtp.toProto(): TConversionSourceSpec.TSFtpSettings {
    val obj = this
    return TConversionSourceSpec.TSFtpSettings.newBuilder().apply {
        host = obj.host
        port = obj.port
        login = obj.login
        encryptedPassword = obj.encryptedPassword
        path = obj.path
    }.build()
}

private fun extractDestination(destinationCase: DestinationCase, spec: TConversionSourceSpec): Destination? {
    return when (destinationCase) {
        DestinationCase.CRM_API_DESTINATION -> spec.crmApiDestination.let {
            Destination.CrmApi(it.counterId, it.accessUid)
        }
        DestinationCase.POST_BACK_DESTINATION -> Destination.PostBack
        else -> {
            LOGGER.warn("unknown conversion destination $destinationCase")
            null
        }
    }
}

private fun extractDestinationCounterIdOrZero(destinationCase: DestinationCase, spec: TConversionSourceSpec): Long {
    return when (destinationCase) {
        DestinationCase.CRM_API_DESTINATION -> spec.crmApiDestination.let {
            it.counterId
        }
        DestinationCase.POST_BACK_DESTINATION -> 0
        else -> {
            LOGGER.warn("unknown conversion destination $destinationCase")
            0
        }
    }
}

private fun Destination.CrmApi.toProto(): TConversionSourceSpec.TCrmApiDestination {
    val obj = this
    return TConversionSourceSpec.TCrmApiDestination.newBuilder()
        .apply {
            counterId = obj.counterId
            accessUid = obj.accessUid
        }.build()
}

private fun Destination.PostBack.toProto(): TPostBackDestination = TPostBackDestination.newBuilder().build()

private fun TProcessingInfo.toCore(): ProcessingInfo {
    return ProcessingInfo(
        processingStatus = EProcessingStatus.forNumber(processingStatus).toCore(),
        lastStartTime = moscowDateTimeFromGrut(lastStartTime),
        lastScheduledTime = moscowDateTimeFromGrut(lastScheduledTime)
    )
}

private fun ProcessingInfo.toProto(): TProcessingInfo {
    val obj = this
    return TProcessingInfo.newBuilder().apply {
        processingStatus = obj.processingStatus.toProto().number
        lastStartTime = obj.lastStartTime?.let { moscowDateTimeToGrut(it) } ?: 0
        lastScheduledTime = obj.lastScheduledTime?.let { moscowDateTimeToGrut(it) } ?: 0
    }.build()
}

private fun EProcessingStatus.toCore(): ProcessingStatus {
    return when (this) {
        EProcessingStatus.PS_NEW -> ProcessingStatus.NEW
        EProcessingStatus.PS_PROCESSING -> ProcessingStatus.PROCESSING
        EProcessingStatus.PS_SUCCESS -> ProcessingStatus.SUCCESS
        EProcessingStatus.PS_ERROR -> ProcessingStatus.ERROR
        else -> ProcessingStatus.NEW
    }
}

private fun ProcessingStatus.toProto(): EProcessingStatus {
    return when (this) {
        ProcessingStatus.NEW -> EProcessingStatus.PS_NEW
        ProcessingStatus.PROCESSING -> EProcessingStatus.PS_PROCESSING
        ProcessingStatus.SUCCESS -> EProcessingStatus.PS_SUCCESS
        ProcessingStatus.ERROR -> EProcessingStatus.PS_ERROR
    }
}
