package ru.yandex.direct.core.grut.api

import java.math.BigDecimal
import java.time.Duration
import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.creative.model.Creative
import ru.yandex.direct.core.entity.creative.model.CreativeType
import ru.yandex.direct.core.entity.creative.model.ModerationInfo
import ru.yandex.direct.core.entity.creative.repository.CreativeConstants.VIDEO_TYPES
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.mysql2grut.enummappers.CreativeEnumMappers
import ru.yandex.direct.mysql2grut.enummappers.CreativeEnumMappers.Companion.sourceMediaTypeToGrut
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Creative.TCreativeSpec
import ru.yandex.grut.objects.proto.client.Schema

data class CreativeGrutModel(
    val creative: Creative,
    val clientId: Long,
)

class CreativeGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<CreativeGrutModel>(grutContext, Schema.EObjectType.OT_CREATIVE, properties) {

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }

    override fun buildIdentity(id: Long): ByteString = Schema.TCreativeMeta.newBuilder()
        .setId(id).build().toByteString()

    override fun parseIdentity(identity: ByteString): Long = Schema.TCreativeMeta.parseFrom(identity).id

    override fun getMetaId(rawMeta: ByteString): Long = Schema.TCreative.parseFrom(rawMeta).meta.id

    override fun serializeMeta(obj: CreativeGrutModel): ByteString {
        return Schema.TCreativeMeta.newBuilder().apply {
            id = obj.creative.id
            clientId = obj.clientId
            creativeType = CreativeEnumMappers.toCreativeType(obj.creative.type).number
        }
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: CreativeGrutModel): ByteString {
        val creative = obj.creative
        return TCreativeSpec.newBuilder().apply {
            creative.name?.let { name = it }
            creative.width?.let { width = it.toInt() }
            creative.height?.let { height = it.toInt() }
            creative.isAdaptive?.let { isAdaptive = it }
            creative.previewUrl?.let { previewUrl = it }
            creative.isBrandLift?.let { isBrandLift = it }
            creative.sourceMediaType?.let { sourceMediaType = sourceMediaTypeToGrut(it).number }
            creative.layoutId?.let { layoutId = it.toInt() }
            creative.livePreviewUrl?.let { livePreviewUrl = it }
            creative.duration?.let { duration = it.toInt() }
            if (creative.type in VIDEO_TYPES) {
                videoData = convertVideoData(creative)
            } else if (creative.type == CreativeType.HTML5_CREATIVE) {
                html5Data = convertHtml5Data(creative)
            }
            if (creative.moderationInfo != null) {
                moderationInfo = convertModerationInfo(creative.moderationInfo)
            }
        }
            .build()
            .toByteString()
    }

    private fun convertHtml5Data(creative: Creative): TCreativeSpec.THtml5 {
        return TCreativeSpec.THtml5.newBuilder().apply {
            creative.isGenerated?.let { isGenerated = it }
            creative.archiveUrl?.let { zipArchiveUrl = it }
            creative.yabsData?.basePath?.let { yabsDataBasePath = it }
            creative.expandedPreviewUrl?.let { expandedPreviewUrl = it }
        }.build()
    }

    private fun convertVideoData(creative: Creative): TCreativeSpec.TVideo {
        return TCreativeSpec.TVideo.newBuilder().apply {
            creative.stockCreativeId?.let { stockCreativeId = it }
            creative.hasPackshot?.let { hasPackshot = it }
            if (creative.additionalData != null) {
                creative.additionalData.duration?.let {
                    duration = it.multiply(BigDecimal.valueOf(1000_000)).toLong()
                }
                if (creative.additionalData.formats != null) {
                    addAllVideoFormats(
                        creative.additionalData.formats.map { format ->
                            TCreativeSpec.TVideo.TVideoFormat.newBuilder().apply {
                                format.width?.let { width = it }
                                format.height?.let { height = it }
                                format.type?.let { type = it }
                                format.url?.let { url = it }
                            }.build()
                        }
                    )
                }
            }
        }.build()
    }

    private fun convertModerationInfo(moderationInfo: ModerationInfo): TCreativeSpec.TModerationInfo {
        return TCreativeSpec.TModerationInfo.newBuilder().apply {
            moderationInfo.contentId?.let { contentId = it }
            moderationInfo.adminRejectReason?.let { adminRejectReason = it }
            moderationInfo.bgrcolor?.let { backgroundColor = it }
            if (moderationInfo.html != null) {
                html = TCreativeSpec.TModerationInfo.THtml.newBuilder().apply {
                    moderationInfo.html.url?.let { url = it }
                }.build()
            }
            if (moderationInfo.images != null) {
                addAllImages(
                    moderationInfo.images.map { image ->
                        TCreativeSpec.TModerationInfo.TImage.newBuilder().apply {
                            image.url?.let { url = it }
                            image.type?.let { type = it }
                            image.originalFileId?.let { originalFileId = it }
                        }.build()
                    }
                )
            }
            if (moderationInfo.texts != null) {
                addAllTexts(
                    moderationInfo.texts.map { textInfo ->
                        TCreativeSpec.TModerationInfo.TText.newBuilder().apply {
                            textInfo.text?.let { text = it }
                            textInfo.type?.let { type = it }
                            textInfo.color?.let { color = it }
                        }.build()
                    }
                )
            }
            if (moderationInfo.videos != null) {
                addAllVideos(
                    moderationInfo.videos.map { video ->
                        TCreativeSpec.TModerationInfo.TVideo.newBuilder().apply {
                            video.stockId?.let { stockId = it }
                            video.url?.let { url = it }
                        }.build()
                    }
                )
            }
            if (moderationInfo.sounds != null) {
                addAllSounds(
                    moderationInfo.sounds.map { sound ->
                        TCreativeSpec.TModerationInfo.TSound.newBuilder().apply {
                            sound.stockId?.let { stockId = it }
                            sound.url?.let { url = it }
                        }.build()
                    }
                )
            }
            if (moderationInfo.aspects != null) {
                addAllAspects(
                    moderationInfo.aspects.map { aspect ->
                        TCreativeSpec.TModerationInfo.TAspect.newBuilder().apply {
                            aspect.width?.let { width = it.toInt() }
                            aspect.height?.let { height = it.toInt() }
                        }.build()
                    }
                )
            }
        }.build()
    }

    fun getCreatives(ids: Collection<Long>): List<Schema.TCreative> {
        val rawObjects = getObjectsByIds(ids)
        return rawObjects.filter { it.protobuf.size() > 0 }.map { transformToCreative(it)!! }
    }

    fun getCreative(id: Long): Schema.TCreative? {
        return getCreatives(listOf(id)).firstOrNull()
    }

    private val updatePaths = listOf("/spec")

    fun createOrUpdateCreatives(objects: List<CreativeGrutModel>) {
        createOrUpdateObjects(objects, updatePaths)
    }

    fun createOrUpdateCreativesParallel(objects: List<CreativeGrutModel>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, updatePaths)
    }

    private fun transformToCreative(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TCreative? {
        if (raw == null) return null
        return Schema.TCreative.parseFrom(raw.protobuf)
    }
}
