package ru.yandex.direct.core.grut.api

import com.google.common.annotations.Beta
import com.google.protobuf.ByteString
import com.google.protobuf.GeneratedMessageV3
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.waitFutures
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TCreatePayload
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqCreateObjects
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqGetObjects
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqRemoveObjects
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqSelectObjects
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TReqUpdateObjects
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TUpdatePayload
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TVersionedPayload
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration

data class VersionedGrutObject<T>(
    val grutObject: T,
    val timestamp: Long,
)

data class UpdatedObject(
    val meta: ByteString,
    val spec: ByteString,
    val setPaths: Collection<String>,
    // служебное поле, запись которого возможна (в качестве исключения) из репликации
    val status: ByteString? = null,
    val removePaths: Collection<String> = listOf(),
)

interface GrutApi<T> {
    fun createObject(obj: T): Long
    fun createObjects(objects: Collection<T>): List<Long>

    /**
     * setPath: какие поля в протобуфе надо будет обновить
     * removePath: какие поля в протобуфе надо будет удалить
     */
    fun createOrUpdateObjects(objects: Collection<T>, setPaths: List<String>, removePaths: List<String> = emptyList())

    fun createOrUpdateObject(
        obj: T,
        setPaths: List<String>,
        removePaths: List<String> = emptyList(),
        createNonExistent: Boolean = true
    )

    /**
     * ignoreNonexistent: если true - игнорировать попытки удалить несуществующий объект, кидать ошибку
     */
    fun deleteObjects(ids: Collection<Long>, ignoreNonexistent: Boolean = true)
    fun deleteObject(id: Long, ignoreNonexistent: Boolean = true) {
        deleteObjects(listOf(id), ignoreNonexistent)
    }

    fun deleteObjectsParallel(ids: Collection<Long>, ignoreNonexistent: Boolean = true, timeout: Duration)

    fun getExistingObjects(ids: Collection<Long>): List<Long>
    fun getExistingObjectsParallel(ids: Collection<Long>, timeout: Duration): List<Long>

    /**
     * ignoreNonexistent: если true - игнорировать попытку получения несуществующего объекта, если false - кидать ошибку
     * skipNonexistent: ecли true - пропускать в ответе несуществующие объекты, если false - вовзращать пустой протобуф
     * для несуществующих объектов (размер списка identities равен размеру списка в ответе)
     * если выставлен ignoreNonExistent = false, то skipNonexistent не имеет смысла, так как для несуществующего объекта
     * будет выброшена ошибка.
     */
    fun getObjects(
        identities: Collection<ByteString>,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        ignoreNonexistent: Boolean = true,
        skipNonexistent: Boolean = true,
        fetchTimestamps: Boolean = false
    ): List<TVersionedPayload>

    fun getObjectsParallel(
        identities: Collection<ByteString>,
        timeout: Duration,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        ignoreNonexistent: Boolean = true,
        skipNonexistent: Boolean = true,
        fetchTimestamps: Boolean = false
    ): List<TVersionedPayload>

    fun getObjectsByIds(
        ids: Collection<Long>,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        ignoreNonexistent: Boolean = true,
        skipNonexistent: Boolean = true
    ): List<TVersionedPayload>

    fun getObjectsByIdsParallel(
        ids: Collection<Long>,
        timeout: Duration,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        ignoreNonexistent: Boolean = true,
        skipNonexistent: Boolean = true
    ): List<TVersionedPayload>

    fun <T> getObjectAs(id: Long, transform: (raw: TVersionedPayload?) -> T?): T?
    fun selectObjects(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        index: String? = null,
        limit: Long? = null,
        continuationToken: String? = null,
        allowFullScan: Boolean = false
    ): List<TVersionedPayload>

    fun <T> selectObjectsAs(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        index: String? = null,
        limit: Long? = null,
        continuationToken: String? = null,
        allowFullScan: Boolean = false,
        transform: (raw: TVersionedPayload?) -> T?
    ): List<T>
}

abstract class GrutApiBase<T>(
    private val grutContext: GrutContext,
    private val type: Schema.EObjectType,
    private val properties: GrutApiProperties
) : GrutApi<T> {
    companion object {
        const val GRUT_GET_OBJECTS_ATTEMPTS = 3
        const val GRUT_CHANGE_OBJECTS_ATTEMPTS = 5
    }

    private val createOrUpdateObjectsBatchSize: Int
        get() {
            return properties.createOrUpdateObjectsBatchSize()
        }

    private val getObjectsBatchSize: Int
        get() {
            return properties.getObjectsBatchSize()
        }

    private val deleteObjectsBatchSize: Int
        get() {
            return properties.deleteObjectsBatchSize()
        }

    override fun selectObjects(
        filter: String,
        attributeSelector: List<String>,
        index: String?,
        limit: Long?,
        continuationToken: String?,
        allowFullScan: Boolean
    ): List<TVersionedPayload> {
        val response =
            grutContext.client.selectObjects(TReqSelectObjects.newBuilder().apply {
                objectType = type
                this.filter = filter
                index?.let { this.index = index }
                limit?.let { this.limit = limit }
                continuationToken?.let { this.continuationToken = continuationToken }
                addAllAttributeSelector(attributeSelector)

                // Index can have hash_expression column for balancing load across shards.
                // Using inequality operators in filter does not allow calculate
                // hash_expression. Therefore, request with index marked as full scan.
                this.allowFullScan = index != null || allowFullScan
            }.build())
        return response.payloadsList
    }

    override fun <T> selectObjectsAs(
        filter: String,
        attributeSelector: List<String>,
        index: String?,
        limit: Long?,
        continuationToken: String?,
        allowFullScan: Boolean,
        transform: (raw: TVersionedPayload?) -> T?
    ): List<T> {
        val payloads = selectObjects(filter, attributeSelector, index, limit, continuationToken, allowFullScan)
        return payloads.mapNotNull { transform(it) }
    }

    override fun deleteObjects(ids: Collection<Long>, ignoreNonexistent: Boolean) {
        val identities = ids.map { buildIdentity(it) }
        deleteObjectsByIdentities(identities, ignoreNonexistent)
    }

    override fun deleteObjectsParallel(ids: Collection<Long>, ignoreNonexistent: Boolean, timeout: Duration) {
        val identities = ids.map { buildIdentity(it) }
        deleteObjectsByIdentitiesParallel(identities, ignoreNonexistent, timeout)
    }

    protected fun deleteObjectsByIdentities(identities: Collection<ByteString>, ignoreNonexistent: Boolean) {
        sequentialProcedure(identities, deleteObjectsBatchSize) { payloadsChunk ->
            grutContext.client.removeObjects(
                TReqRemoveObjects.newBuilder().apply {
                    objectType = type
                    addAllIdentities(payloadsChunk)
                    this.ignoreNonexistent = ignoreNonexistent
                }.build()
            )
        }
    }

    protected fun deleteObjectsByIdentitiesParallel(
        identities: Collection<ByteString>,
        ignoreNonexistent: Boolean,
        timeout: Duration
    ) {
        if (identities.isEmpty()) {
            return
        }

        val futures = identities.chunked(deleteObjectsBatchSize) { payloadsChunk ->
            grutContext.client.removeObjectsAsync(
                TReqRemoveObjects.newBuilder().apply {
                    objectType = type
                    addAllIdentities(payloadsChunk)
                    this.ignoreNonexistent = ignoreNonexistent
                }.build()
            )
        }.toTypedArray()

        waitFutures(futures, timeout)
    }

    /**
     * Принимает список идентификаторов объектов
     * Возвращает идентификаторы существующих объектов
     */
    override fun getExistingObjects(ids: Collection<Long>): List<Long> {
        val payloads = getObjectsByIds(ids, attributeSelector = listOf("/meta/id"), skipNonexistent = true)
        return payloads.map { getMetaId(it.protobuf) }
    }

    /**
     * Принимает список идентификаторов объектов
     * Возвращает идентификаторы существующих объектов
     */
    override fun getExistingObjectsParallel(ids: Collection<Long>, timeout: Duration): List<Long> {
        val payloads =
            getObjectsByIdsParallel(ids, timeout, attributeSelector = listOf("/meta/id"), skipNonexistent = true)
        return payloads.map { getMetaId(it.protobuf) }
    }

    override fun createObject(obj: T): Long {
        return createObjects(listOf(obj))[0]
    }

    override fun createObjects(objects: Collection<T>): List<Long> {
        if (objects.isEmpty()) {
            return listOf()
        }
        val payloads = objects.map {
            TCreatePayload.newBuilder().apply {
                meta = serializeMeta(it)
                spec = serializeSpec(it)
                serializeStatus(it)?.also {
                    status = it
                }
            }.build()
        }
        val result = sequential(payloads, createOrUpdateObjectsBatchSize) { payloadsChunk ->
            grutContext.client.createObjects(
                TReqCreateObjects.newBuilder().apply {
                    objectType = type
                    addAllPayloads(payloadsChunk)
                }.build()
            ).identitiesList
                .map { parseIdentity(it) }
        }
        return result
    }

    /**
     * Разбивает вызов одного запроса с большим количеством параметров на несколько запросов с количеством параметров
     * не превосходящих batchSize
     * Объединяет списки ответов в один список, как будто это был вызов одного метода
     */
    private fun <T, R> sequential(
        allPayloads: Collection<T>,
        batchSize: Int,
        request: (payloads: List<T>) -> Collection<R>
    ): List<R> {
        val chunks = allPayloads.chunked(batchSize)
        val result: MutableList<R> = mutableListOf()
        for (chunk in chunks) {
            result.addAll(request(chunk))
        }
        return result
    }

    /**
     * Разбивает вызов одного запроса с большим количеством параметров на несколько запросов с количеством параметров
     * не превосходящих batchSize
     * Ответ не возвращает
     * */
    private fun <T, R> sequentialProcedure(
        allPayloads: Collection<T>,
        batchSize: Int,
        request: (payloads: List<T>) -> R
    ) {
        val chunks = allPayloads.chunked(batchSize)
        for (chunk in chunks) {
            request(chunk)
        }
    }

    override fun createOrUpdateObjects(
        objects: Collection<T>,
        setPaths: List<String>,
        removePaths: List<String>,
    ) {
        val payloads = prepareObjectsForUpdate(objects, setPaths, removePaths)
        return createOrUpdateObjects(payloads)
    }

    /**
     * Обновляет чанки асинхронно
     */
    fun createOrUpdateObjectsParallel(
        objects: Collection<T>,
        timeout: Duration,
        setPaths: List<String>,
        removePaths: List<String> = listOf(),
    ) {
        val payloads = prepareObjectsForUpdate(objects, setPaths, removePaths)
        return createOrUpdateObjectsParallel(payloads, timeout)
    }

    private fun internalCreateOrUpdateObjects(
        objects: Collection<UpdatedObject>,
        createNonExistent: Boolean,
        ignoreNonExistent: Boolean? = null
    ) {
        if (objects.isEmpty()) {
            return
        }
        val payloads = preparePayloads(objects)
        sequential(payloads, createOrUpdateObjectsBatchSize) { payloadsChunk ->
            grutContext.client.updateObjects(
                TReqUpdateObjects.newBuilder().apply {
                    objectType = type
                    createNonexistent = createNonExistent
                    ignoreNonExistent?.let { ignoreNonexistent = it }
                    addAllPayloads(payloadsChunk)
                }.build()
            ).identitiesList
        }
    }

    /**
     * Выполняет createOrUpdate для каждой пачки размера createOrUpdateObjectsBatchSize асинхронно
     */
    private fun internalCreateOrUpdateObjectsParallel(
        objects: Collection<UpdatedObject>,
        timeout: Duration,
        createNonExistent: Boolean,
        ignoreNonExistent: Boolean? = null
    ) {
        if (objects.isEmpty()) {
            return
        }

        val payloads = preparePayloads(objects)
        val futures = payloads.chunked(createOrUpdateObjectsBatchSize) { payloadsChunk ->
            grutContext.client.updateObjectsAsync(
                TReqUpdateObjects.newBuilder().apply {
                    objectType = type
                    createNonexistent = createNonExistent
                    ignoreNonExistent?.let { ignoreNonexistent = it }
                    addAllPayloads(payloadsChunk)
                }.build()
            )
        }.toTypedArray()

        waitFutures(futures, timeout)
    }

    private fun preparePayloads(objects: Collection<UpdatedObject>) =
        objects.map {
            TUpdatePayload.newBuilder().apply {
                meta = it.meta
                spec = it.spec
                if (it.status != null) {
                    status = it.status
                }
                if (it.setPaths.isNotEmpty()) {
                    addAllSetPaths(it.setPaths)
                    recursive = true
                }
                if (it.removePaths.isNotEmpty()) {
                    addAllRemovePaths(it.removePaths)
                    forceRemoveAttribute = true
                }
            }.build()
        }

    private fun prepareObjectsForUpdate(
        objects: Collection<T>,
        setPaths: List<String>,
        removePaths: List<String>,
    ): Collection<UpdatedObject> {
        val withStatusChanging = hasStatusInPath(setPaths)
        return objects.map {
            UpdatedObject(
                meta = serializeMeta(it),
                spec = serializeSpec(it),
                status = if (withStatusChanging) serializeStatus(it) else null,
                setPaths = setPaths,
                removePaths = removePaths,
            )
        }
    }

    /**
     * Возвращает true если в path явно указано обновление поля в status
     */
    private fun hasStatusInPath(setPaths: List<String>): Boolean = setPaths.any { it.startsWith("/status/") }

    override fun createOrUpdateObject(
        obj: T,
        setPaths: List<String>,
        removePaths: List<String>,
        createNonExistent: Boolean
    ) =
        createOrUpdateObjects(listOf(obj), setPaths, removePaths)

    override fun getObjectsByIds(
        ids: Collection<Long>,
        attributeSelector: List<String>,
        ignoreNonexistent: Boolean,
        skipNonexistent: Boolean,
    ): List<TVersionedPayload> {
        val identities = ids.map { buildIdentity(it) }
        return getObjects(identities, attributeSelector, ignoreNonexistent, skipNonexistent)
    }

    override fun getObjectsByIdsParallel(
        ids: Collection<Long>,
        timeout: Duration,
        attributeSelector: List<String>,
        ignoreNonexistent: Boolean,
        skipNonexistent: Boolean,
    ): List<TVersionedPayload> {
        val identities = ids.map { buildIdentity(it) }
        return getObjectsParallel(identities, timeout, attributeSelector, ignoreNonexistent, skipNonexistent)
    }

    override fun getObjectsParallel(
        identities: Collection<ByteString>,
        timeout: Duration,
        attributeSelector: List<String>,
        ignoreNonexistent: Boolean,
        skipNonexistent: Boolean,
        fetchTimestamps: Boolean
    ): List<TVersionedPayload> {
        val futures = identities.chunked(getObjectsBatchSize) { identitiesChunk ->
            val req =
                getTReqObjects(identitiesChunk, attributeSelector, ignoreNonexistent, skipNonexistent, fetchTimestamps)
            grutContext.client.getObjectsAsync(req)
                .thenApply { it.payloadsList }
        }.toTypedArray()
        waitFutures(futures, timeout)

        return futures.flatMap {
            it.get()
        }
    }

    override fun getObjects(
        identities: Collection<ByteString>,
        attributeSelector: List<String>,
        ignoreNonexistent: Boolean,
        skipNonexistent: Boolean,
        fetchTimestamps: Boolean,
    ): List<TVersionedPayload> {
        val result = sequential(identities, getObjectsBatchSize) { identitiesChunk ->
            val req =
                getTReqObjects(identitiesChunk, attributeSelector, ignoreNonexistent, skipNonexistent, fetchTimestamps)
            grutContext.client.getObjects(req).payloadsList
        }
        return result
    }

    fun getObject(id: Long): TVersionedPayload? {
        return getObjectsByIds(listOf(id)).firstOrNull()
    }

    override fun <T> getObjectAs(id: Long, transform: (raw: TVersionedPayload?) -> T?): T? {
        return transform(getObject(id))
    }

    /**
     * Для возможности создания/обновления частичных моделей, с указанием set/remove path
     */
    protected fun createOrUpdateObjects(objects: Collection<UpdatedObject>) =
        internalCreateOrUpdateObjects(objects, createNonExistent = true)

    /**
     * Этот метод пока нельзя использовать, так как он создан экспериментально для проверки работоспособности асинхронных методов
     */
    @Beta
    protected fun createOrUpdateObjectsParallel(objects: Collection<UpdatedObject>, timeout: Duration) =
        internalCreateOrUpdateObjectsParallel(objects, timeout, createNonExistent = true)

    /**
     * Сейчас нет удобного способа частичного обновления на модели api
     * поэтому, чтобы всё-таки поддержать такое обновление, сделаем явное указание meta, spec, status
     *
     * при попытке обновить несуществующий объект произойдет ошибка
     */
    protected fun updateObjects(objects: Collection<UpdatedObject>) =
        internalCreateOrUpdateObjects(objects, createNonExistent = false, ignoreNonExistent = true)

    protected fun updateObjects(
        objects: Collection<T>,
        setPaths: List<String>,
        removePaths: List<String>,
    ) {
        val payloads = prepareObjectsForUpdate(objects, setPaths, removePaths)
        return updateObjects(payloads)
    }

    protected fun prepareSetRemovePaths(
        proto: GeneratedMessageV3,
        prefix: String,
        fields: List<String>
    ): SetRemovePaths {
        // TODO: когда понадобится — переделать метод на рекурсивный для обхода вглубь
        val container = SetRemovePaths()

        val descriptor = proto.descriptorForType
        for (fieldName in fields) {
            val path = "$prefix/$fieldName"
            val field = descriptor.findFieldByName(fieldName)
            if (field.isRepeated) {
                when (proto.getRepeatedFieldCount(field)) {
                    0 -> container.pathsToRemove.add(path)
                    else -> container.pathsToSet.add(path)
                }
            } else if (proto.hasField(field)) {
                container.pathsToSet.add(path)
            } else {
                container.pathsToRemove.add(path)
            }
        }
        return container
    }

    protected abstract fun buildIdentity(id: Long): ByteString
    protected abstract fun parseIdentity(identity: ByteString): Long
    protected abstract fun serializeMeta(obj: T): ByteString
    protected abstract fun serializeSpec(obj: T): ByteString
    protected open fun serializeStatus(obj: T): ByteString? = null

    protected abstract fun getMetaId(rawMeta: ByteString): Long

    protected data class SetRemovePaths(
        val pathsToSet: MutableSet<String> = mutableSetOf(),
        val pathsToRemove: MutableSet<String> = mutableSetOf(),
    )

    private fun getTReqObjects(
        identities: Collection<ByteString>,
        attributeSelector: List<String>,
        ignoreNonexistent: Boolean,
        skipNonexistent: Boolean,
        fetchTimestamps: Boolean
    ) =
        TReqGetObjects.newBuilder().apply {
            objectType = type
            addAllIdentities(identities)
            this.ignoreNonexistent = ignoreNonexistent
            this.skipNonexistent = skipNonexistent
            this.fetchTimestamps = fetchTimestamps
            if (attributeSelector.isNotEmpty()) {
                this.addAllAttributeSelector(attributeSelector)
            }
        }.build()
}
