package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.grut.model.GrutHistoryEventPage
import ru.yandex.grut.client.GrutClient
import ru.yandex.grut.history.History
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.client.Schema

/**
GrUT фиксирует снапшот состояния записи в момент изменения, с указанием времени, типа события и его автора.
Результат выдачи сортируется по типу объекта, его ID, UUID(в общем случае  1-to-1 с ID),
времени изменения и ID транзакции
 */
abstract class GrutHistoryBaseApi(
    private val grutClient: GrutClient,
    private val type: Schema.EObjectType,
) {

    /**
     * Метод постранично возвращает список событий для указанной записи(историю изменений)
     * в заданном интервале времени
     *
     * @param id - идентификатор записи в формате ByteString
     * @param interval - интервал времени, в котором осуществляется поиск изменений.
     * @param continuationToken - токен для получения следующей страницы запроса, для 0 страницы = null
     * @param limit - размер страницы
     * @param attributeSelector - набор селекторов записи-родителя, которые требуется получить по запросу
     * @param descending - направление сортировки результатов
     * @param distinct - гарантирует уникальность записей в результате
     *
     * @return набор событий(изменений) + токен для чтения следующей страницы
     */
    fun selectObjectsHistory(
        id: ByteString,
        interval: History.TTimeInterval? = null,
        continuationToken: String? = null,
        limit: Int? = null,
        attributeSelector: List<String> = listOf("/meta", "/spec"),
        descending: Boolean = true,
        distinct: Boolean = true
    ): GrutHistoryEventPage {

        val result = grutClient.selectObjectHistory(
            ObjectApiServiceOuterClass.TReqSelectObjectHistory.newBuilder().apply {
                this.identity = id
                this.objectType = type
                interval?.let { this.interval = it }
                limit?.let { this.limit = it }
                continuationToken?.let { this.continuationToken = it }
                addAllAttributeSelector(attributeSelector)
                this.descendingTimeOrder = descending
                this.distinct = distinct
            }.build()
        )
        return GrutHistoryEventPage(result.eventsList, result.continuationToken)
    }
}
