package ru.yandex.direct.core.grut.api

import java.time.Duration
import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.mobilecontent.model.MobileContent
import ru.yandex.direct.core.entity.mobilecontent.model.OsType
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.MobileContent.TMobileContentSpec
import ru.yandex.grut.objects.proto.MobilePlatform
import ru.yandex.grut.objects.proto.client.Schema


class MobileContentGrutApi(grutContext: GrutContext, properies: GrutApiProperties)
    : GrutApiBase<MobileContent>(grutContext, Schema.EObjectType.OT_MOBILE_CONTENT, properies) {
    override fun buildIdentity(id: Long): ByteString {
        return Schema.TMobileContentMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TMobileContentMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: MobileContent): ByteString {
        return Schema.TMobileContentMeta.newBuilder().apply {
            obj.id?.let { id = it }
            clientId = obj.clientId
        }.build().toByteString()
    }

    override fun serializeSpec(obj: MobileContent): ByteString {
        return TMobileContentSpec.newBuilder().apply {
            obj.name?.let { this.name = it }
            obj.bundleId?.let { this.bundleId = it }
            obj.storeContentId?.let { this.storeContentId = it }
            obj.minOsVersion?.let { this.minOsVersion = it }
            obj.osType?.let { this.platform = convertOsType(it).number }
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TMobileContent.parseFrom(rawMeta).meta.id
    }

    fun getMobileContent(id: Long): Schema.TMobileContent? {
        return getObjectAs(id, ::transformToProto)
    }

    val spec = listOf("/spec")
    fun createOrUpdateMobileContent(objects: Collection<MobileContent>) {
        createOrUpdateObjects(objects, spec)
    }

    fun createOrUpdateMobileContentParallel(objects: Collection<MobileContent>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, spec)
    }

    private fun transformToProto(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TMobileContent? {
        if (raw == null) return null
        return Schema.TMobileContent.parseFrom(raw.protobuf)
    }

    companion object {
        fun convertOsType(osType: OsType): MobilePlatform.EMobilePlatform {
            return if (osType == OsType.ANDROID) {
                MobilePlatform.EMobilePlatform.MP_ANDROID
            } else {
                MobilePlatform.EMobilePlatform.MP_IOS
            }
        }

        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }
}
