package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.moneyFromGrut
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeFromGrut
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeToGrut
import ru.yandex.direct.core.grut.api.utils.toCurrency
import ru.yandex.direct.core.grut.api.utils.toGrutMoney
import ru.yandex.direct.core.grut.api.utils.toIsoCode
import ru.yandex.direct.core.grut.model.FeedOfferGrut
import ru.yandex.direct.core.grut.model.FeedOfferParams
import ru.yandex.direct.core.grut.model.FeedOfferPrice
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Offer
import ru.yandex.grut.objects.proto.client.Schema
import java.time.LocalDateTime

class OfferGrutApi(grutContext: GrutContext, properties: GrutApiProperties) :
    GrutApiBase<FeedOfferGrut>(grutContext, Schema.EObjectType.OT_OFFER, properties) {

    companion object {
        const val OFFERS_FETCH_LIMIT = 2000L
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TOfferMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TOfferMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: FeedOfferGrut): ByteString {
        return Schema.TOfferMeta.newBuilder()
            .apply {
                if (obj.id != null && obj.id > 0) {
                    id = obj.id
                }

                clientId = obj.clientId
            }
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: FeedOfferGrut): ByteString {
        return Offer.TOfferSpec.newBuilder()
            .apply {
                label = obj.label
                description = obj.description
                href = obj.href

                if (obj.images.isNotEmpty()) {
                    addAllImages(obj.images)
                }

                currencyIsoCode = toIsoCode(obj.currencyCode)
                price = getOfferPrice(obj.price)
                isAvailable = obj.isAvailable
                updateTime = moscowDateTimeToGrut(obj.updateTime ?: LocalDateTime.now())

                setOfferParams(obj.offerParams)
            }
            .build()
            .toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TOffer.parseFrom(rawMeta).meta.id
    }

    fun getOffer(id: Long): FeedOfferGrut? {
        return getObjectAs(id, ::transformToOffer)
    }

    fun getOfferByIds(ids: Collection<Long>): List<FeedOfferGrut> {
        val rawOffers = getObjectsByIds(ids)
        return rawOffers.filter { it.protobuf.size() > 0 }
            .mapNotNull { transformToOffer(it) }
    }

    fun selectOffersByClientId(clientId: Long): List<FeedOfferGrut> {
        return selectObjectsAs("[/meta/client_id] = $clientId", limit = OFFERS_FETCH_LIMIT, transform = ::transformToOffer)
    }

    fun createOffers(offers: List<FeedOfferGrut>): List<Long> {
        return createObjects(offers)
    }

    private val setPaths = listOf("/spec")
    fun createOrUpdateOffer(offer: FeedOfferGrut) {
        createOrUpdateObject(offer, setPaths)
    }

    fun createOrUpdateOffers(offers: List<FeedOfferGrut>) {
        createOrUpdateObjects(offers, setPaths)
    }

    fun deleteOffers(ids: Collection<Long>) {
        deleteObjects(ids, true);
    }

    private fun transformToOffer(raw: ObjectApiServiceOuterClass.TVersionedPayload?): FeedOfferGrut? {
        if (raw == null) return null
        val parsedOffer = Schema.TOffer.parseFrom(raw.protobuf) ?: return null
        return convertToFeedOffer(parsedOffer)
    }

    private fun convertToFeedOffer(offer: Schema.TOffer): FeedOfferGrut {
        val meta = offer.meta
        val spec = offer.spec

        return FeedOfferGrut(
            id = meta.id,
            clientId = meta.clientId,
            label = spec.label,
            description = spec.description,
            href = spec.href,
            images = spec.imagesList,
            currencyCode = toCurrency(spec.currencyIsoCode),
            price = FeedOfferPrice(
                currentPrice = moneyFromGrut(spec.price.currentPrice),
                oldPrice = if (spec.price.hasOldPrice()) moneyFromGrut(spec.price.oldPrice) else null
            ),
            isAvailable = spec.isAvailable,
            updateTime = moscowDateTimeFromGrut(spec.updateTime),
            offerParams = convertToOfferParams(spec)
        )
    }

    private fun convertToOfferParams(spec: Offer.TOfferSpec): FeedOfferParams {
        return when (spec.offerParamsCase) {
            Offer.TOfferSpec.OfferParamsCase.RETAIL -> {
                val retail = spec.retail

                FeedOfferParams.Retail(
                    category = if (retail.hasCategory()) retail.category else null,
                    vendor = if (retail.hasVendor()) retail.vendor else null,
                    model = if (retail.hasModel()) retail.model else null
                )
            }
            null, Offer.TOfferSpec.OfferParamsCase.OFFERPARAMS_NOT_SET ->
                throw RuntimeException("Offer params are not specified")
        }
    }

    private fun getOfferPrice(feedOfferPrice: FeedOfferPrice): Offer.TOfferPrice {
        return Offer.TOfferPrice.newBuilder()
            .apply {
                currentPrice = toGrutMoney(feedOfferPrice.currentPrice)
                feedOfferPrice.oldPrice?.let { oldPrice = toGrutMoney(it) }
            }
            .build()
    }

    private fun Offer.TOfferSpec.Builder.setOfferParams(offerParams: FeedOfferParams) {
        when (offerParams) {
            is FeedOfferParams.Retail ->
                retail = Offer.TRetailOffer.newBuilder().apply {
                    offerParams.category?.let { category = offerParams.category }
                    offerParams.vendor?.let { vendor = offerParams.vendor }
                    offerParams.model?.let { model = offerParams.model }
                }.build()
            else -> throw RuntimeException("Unknown offer params type: $offerParams")
        }
    }
}
