package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.mysql2grut.enummappers.AdGroupEnumMappers.Companion.toGrutAdGroupType
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Package
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration

data class PricePackageGrut(
    val id: Long,
    val auctionPriority: Long?,
    val allowedDomains: List<String>? = null,
    val adGroupTypes: Set<AdGroupType>? = null,
)

class PricePackageGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<PricePackageGrut>(grutContext, Schema.EObjectType.OT_PACKAGE, properties) {
    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TPackageMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TPackageMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: PricePackageGrut): ByteString {
        return Schema.TPackageMeta.newBuilder()
            .apply {
                id = obj.id
            }
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: PricePackageGrut): ByteString {
        return Package.TPackageSpec.newBuilder().apply {
            auctionPriority = obj.auctionPriority?.toInt() ?: 0
            addAllAllowedDomains(obj.allowedDomains ?: listOf())
            addAllAvailableAdGroupTypes(obj.adGroupTypes?.map { toGrutAdGroupType(it).number } ?: listOf())
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TPackage.parseFrom(rawMeta).meta.id
    }

    fun getPricePackages(ids: Collection<Long>): List<Schema.TPackage> {
        val rawObjects = getObjectsByIds(ids)
        return rawObjects.filter { it.protobuf.size() > 0 }.map { transformToPricePackage(it)!! }
    }

    fun getPricePackage(id: Long): Schema.TPackage? {
        return getPricePackages(listOf(id)).firstOrNull()
    }

    private val updatePaths = listOf("/spec")

    fun createOrUpdatePackages(objects: List<PricePackageGrut>) {
        createOrUpdateObjects(objects, updatePaths)
    }

    fun createOrUpdatePackagesParallel(objects: List<PricePackageGrut>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, updatePaths)
    }

    private fun transformToPricePackage(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TPackage? {
        if (raw == null) return null
        return Schema.TPackage.parseFrom(raw.protobuf)
    }
}
