package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import org.slf4j.LoggerFactory
import ru.yandex.direct.core.entity.retargeting.model.ConditionType
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition
import ru.yandex.direct.core.entity.retargeting.model.Rule
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.mysql2grut.repository.RetargetingGoal
import ru.yandex.direct.mysql2grut.enummappers.RetargetingConditionEnumMappers.Mappers.toGrut
import ru.yandex.direct.mysql2grut.enummappers.RetargetingConditionEnumMappers.Mappers.toGrutGoalSource
import ru.yandex.grut.auxiliary.proto.RetargetingRule.TRetargetingRule
import ru.yandex.grut.auxiliary.proto.RetargetingRule.TRetargetingRule.TABBidModifierCondition
import ru.yandex.grut.auxiliary.proto.RetargetingRule.TRetargetingRule.TABExperiment
import ru.yandex.grut.auxiliary.proto.RetargetingRule.TRetargetingRule.TABExperimentList
import ru.yandex.grut.auxiliary.proto.RetargetingRule.TRetargetingRule.TGoal
import ru.yandex.grut.auxiliary.proto.RetargetingRule.TRetargetingRule.TGoalCombination
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TVersionedPayload
import ru.yandex.grut.objects.proto.RetargetingCondition.TRetargetingConditionSpec
import ru.yandex.grut.objects.proto.client.Schema.EObjectType.OT_RETARGETING_CONDITION
import ru.yandex.grut.objects.proto.client.Schema.TRetargetingCondition
import ru.yandex.grut.objects.proto.client.Schema.TRetargetingConditionMeta
import java.time.Duration
import java.util.Optional

data class RetargetingConditionGrut(
    val retCond: RetargetingCondition,
    val goals: Map<Long, RetargetingGoal>
)

class RetargetingConditionGrutApi(grutContext: GrutContext, properties: GrutApiProperties) :
    GrutApiBase<RetargetingConditionGrut>(grutContext, OT_RETARGETING_CONDITION, properties) {

    override fun buildIdentity(id: Long): ByteString {
        return TRetargetingConditionMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return TRetargetingConditionMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: RetargetingConditionGrut): ByteString {
        return TRetargetingConditionMeta.newBuilder()
            .setId(obj.retCond.id)
            .setClientId(obj.retCond.clientId)
            .setRetargetingConditionType(toGrut(obj.retCond.type).number)
            .build()
            .toByteString()
    }

    fun Rule.getValidGoals(): List<Goal> {
        return this.goals.filter { it.id != null }
    }

    override fun serializeSpec(obj: RetargetingConditionGrut): ByteString {
        return TRetargetingConditionSpec.newBuilder().apply {
            if (obj.retCond.name != null) {
                this.name = obj.retCond.name
            }
            if (obj.retCond.description != null) {
                this.description = obj.retCond.description
            }

            // DIRECT-174193 Реплицируем ret_cond пустым retargeting_json в MySQL,
            // чтобы создались внешние ключи в кампаниях.
            // но не заполняем секцию /spec/rule для таких объектов.
            val validGoals = obj.retCond.collectGoalsSafe().filter { it.id != null }
            if (validGoals.isNotEmpty()) {
                this.rule = TRetargetingRule.newBuilder().apply {
                    if (obj.retCond.type == ConditionType.ab_segments) {
                        // Пока нет четкого критерия, какие ab-условия относятся к экспериментам, какие к корректировкам
                        // Заполняю оба поля
                        bidModifierAbCondition = TABBidModifierCondition.newBuilder()
                            .apply {
                                goal = mapGoal(obj.retCond.collectGoals().first(), obj.goals)
                                sectionId = obj.retCond.rules.first().sectionId
                            }
                            .build()
                        val items = obj.retCond.rules.map {
                            val goals = it.getValidGoals().map { g -> mapGoal(g, obj.goals) }
                            TABExperiment.newBuilder().apply {
                                addAllGoals(goals)
                                operator = toGrut(it.type).number
                                sectionId = it.sectionId
                            }.build()
                        }
                        abExperiments = TABExperimentList.newBuilder().addAllItems(items).build()
                    } else {
                        val combinations = obj.retCond.rules.map {
                            TGoalCombination.newBuilder().apply {
                                operator = toGrut(it.type).number
                                val goals = it.getValidGoals().map { g -> mapGoal(g, obj.goals) }
                                addAllGoals(goals)
                            }.build()
                        }
                        goalCombinations = TRetargetingRule
                            .TGoalCombinationList.newBuilder()
                            .addAllItems(combinations)
                            .build()
                    }
                }.build()
            }

            this.isDeleted = obj.retCond.deleted
        }
            .build()
            .toByteString()
    }

    private fun mapGoal(g: Goal, extraGoalInfo: Map<Long, RetargetingGoal>): TGoal {
        val isAccessible = when {
            extraGoalInfo.containsKey(g.id) -> extraGoalInfo[g.id]!!.isAccessible
            else -> false
        }

        return TGoal.newBuilder()
            .setGoalId(g.id)
            .setGoalSource(toGrutGoalSource(g.type).number)
            .setGoalType(toGrut(g.type).number)
            .setIsAccessible(isAccessible)
            .setValidDays(Optional.ofNullable(g.time).orElse(0).toLong())
            .build()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return TRetargetingCondition.parseFrom(rawMeta).meta.id
    }

    private val setPaths = listOf("/spec")
    fun createOrUpdateRetargetingConditions(retargetingConditions: List<RetargetingConditionGrut>) {
        createOrUpdateObjects(retargetingConditions, setPaths)
    }

    fun createOrUpdateRetargetingConditionsParallel(retargetingConditions: List<RetargetingConditionGrut>) {
        createOrUpdateObjectsParallel(retargetingConditions, UPDATE_TIMEOUT, setPaths)
    }

    fun getRetargetingCondition(id: Long): TRetargetingCondition? {
        return getObjectAs(id, ::transformToRetCond)
    }

    fun getRetargetingConditions(
        ids: List<Long>,
        attrs: List<String> = listOf("/meta", "/spec", "/status")
    ): List<TRetargetingCondition> {
        val rawObjects = getObjectsByIds(ids, attrs)
        return rawObjects.filter { it.protobuf.size() > 0 }.map { transformToRetCond(it)!! }
    }

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)

        private fun transformToRetCond(raw: TVersionedPayload?): TRetargetingCondition? {
            if (raw == null) return null
            return TRetargetingCondition.parseFrom(raw.protobuf)
        }

        val logger = LoggerFactory.getLogger(RetargetingConditionGrutApi::class.java)
    }
}

