package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Strategy
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration

data class StrategyGrut(
    val id: Long,
    val clientId: Long,
    val name: String?,
)

class StrategyGrutApi(grutContext: GrutContext, properties: GrutApiProperties) :
    GrutApiBase<StrategyGrut>(grutContext, Schema.EObjectType.OT_STRATEGY, properties) {
    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TStrategyMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TStrategyMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: StrategyGrut): ByteString {
        return Schema.TStrategyMeta.newBuilder()
            .apply {
                id = obj.id
                clientId = obj.clientId
            }
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: StrategyGrut): ByteString {
        return Strategy.TStrategySpec.newBuilder().apply {
            name = obj.name ?: ""
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TStrategy.parseFrom(rawMeta).meta.id
    }

    fun getStrategies(ids: Collection<Long>): List<Schema.TStrategy> {
        val rawObjects = getObjectsByIds(ids)
        return rawObjects.filter { it.protobuf.size() > 0 }.map { transformToStrategy(it)!! }
    }

    fun getStrategy(id: Long): Schema.TStrategy? {
        return getStrategies(listOf(id)).firstOrNull()
    }

    private val updatePaths = listOf("/spec")

    fun createOrUpdateStrategies(objects: List<StrategyGrut>) {
        createOrUpdateObjects(objects, updatePaths)
    }

    fun createOrUpdateStrategiesParallel(objects: List<StrategyGrut>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, updatePaths)
    }

    private fun transformToStrategy(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TStrategy? {
        if (raw == null) return null
        return Schema.TStrategy.parseFrom(raw.protobuf)
    }
}
