package ru.yandex.direct.core.grut.api

import java.time.Duration
import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.vcard.model.PointOnMap
import ru.yandex.direct.core.entity.vcard.model.Vcard
import ru.yandex.direct.core.entity.vcard.repository.VcardMappings.phoneToDb
import ru.yandex.direct.core.entity.vcard.repository.internal.DbAddress
import ru.yandex.direct.core.grut.api.utils.bigDecimalToGrut
import ru.yandex.direct.mysql2grut.enummappers.VCardEnumMappers.Companion.precisionTypeToGrut
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Vcard.TVCardSpec
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.objects.proto.client.Schema.TVCard

data class VCardGrutModel(
    val vcard: Vcard,
    val clientId: Long,
    val address: DbAddress?,
    val mapPoint: PointOnMap?,
    val mapPointAuto: PointOnMap?,
)

class VCardGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<VCardGrutModel>(grutContext, Schema.EObjectType.OT_VCARD, properties) {

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
    }

    override fun buildIdentity(id: Long): ByteString = Schema.TVCardMeta.newBuilder()
        .setId(id).build().toByteString()

    override fun parseIdentity(identity: ByteString): Long = Schema.TVCardMeta.parseFrom(identity).id

    override fun getMetaId(rawMeta: ByteString): Long = TVCard.parseFrom(rawMeta).meta.id

    override fun serializeMeta(obj: VCardGrutModel): ByteString {
        return Schema.TVCardMeta.newBuilder().apply {
            id = obj.vcard.id
            clientId = obj.clientId
        }
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: VCardGrutModel): ByteString {
        val vcard = obj.vcard
        return TVCardSpec.newBuilder().apply {
            vcard.geoId?.let { geoId = it }
            vcard.phone?.let { phone = phoneToDb(it) }
            vcard.companyName?.let { name = it }
            vcard.city?.let { city = it }
            vcard.contactPerson?.let { contactPerson = it }
            vcard.workTime?.let { worktime = it }
            vcard.country?.let { country = it }
            vcard.street?.let { street = it }
            vcard.house?.let { house = it }
            vcard.build?.let { build = it }
            vcard.apart?.let { apart = it }
            vcard.metroId?.let { metroId = it }
            vcard.extraMessage?.let { extraMessage = it }
            vcard.email?.let { contactEmail = it }
            vcard.instantMessenger?.type?.let { imClient = it }
            vcard.instantMessenger?.login?.let { imLogin = it }
            vcard.ogrn?.let { ogrn = it }
            vcard.permalink?.let { permalinkId = it }
            obj.address?.let {
                address = TVCardSpec.TAddress.newBuilder().apply {
                    obj.mapPoint?.let { mapPoint = pointToGrut(it) }
                    obj.mapPointAuto?.let { mapPointAuto = pointToGrut(it) }
                    obj.address.address?.let { address = it }
                    obj.address.metroId?.let { metroId = it }
                    obj.address.precision?.let { precisionType = precisionTypeToGrut(it).number }
                }
                    .build()
            }
        }
            .build()
            .toByteString()
    }

    private fun pointToGrut(mapId: PointOnMap): TVCardSpec.TAddress.TMapPoint {
        return TVCardSpec.TAddress.TMapPoint.newBuilder().apply {
            mapId.x?.let { lon = bigDecimalToGrut(it) }
            mapId.y?.let { lat = bigDecimalToGrut(it) }
            mapId.x1?.let { lowerCornerLon = bigDecimalToGrut(it) }
            mapId.y1?.let { lowerCornerLat = bigDecimalToGrut(it) }
            mapId.x2?.let { upperCornerLon = bigDecimalToGrut(it) }
            mapId.y2?.let { upperCornerLat = bigDecimalToGrut(it) }
        }.build()
    }

    fun getVCards(ids: Collection<Long>): List<TVCard> {
        val rawObjects = getObjectsByIds(ids)
        return rawObjects.filter { it.protobuf.size() > 0 }.map { transformToVCard(it)!! }
    }

    private val updatePaths = listOf("/spec")

    fun createOrUpdateVCards(objects: List<VCardGrutModel>) {
        createOrUpdateObjects(objects, updatePaths)
    }

    fun createOrUpdateVCardsParallel(objects: List<VCardGrutModel>) {
        createOrUpdateObjectsParallel(objects, UPDATE_TIMEOUT, updatePaths)
    }

    private fun transformToVCard(raw: ObjectApiServiceOuterClass.TVersionedPayload?): TVCard? {
        if (raw == null) return null
        return TVCard.parseFrom(raw.protobuf)
    }
}
