package ru.yandex.direct.core.grut.api.utils

import java.math.BigDecimal
import java.math.RoundingMode
import java.time.Instant
import java.time.LocalDate
import java.time.LocalDateTime
import java.time.OffsetTime
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.direct.utils.DateTimeUtils.MSK

private val MONEY_MULTIPLIER = BigDecimal.valueOf(1000000)

fun bigDecimalToGrut(money: BigDecimal): Long =
    money.multiply(MONEY_MULTIPLIER).setScale(0, RoundingMode.HALF_UP).toLong()

fun bigDecimalFromGrut(money: Long): BigDecimal =
    BigDecimal.valueOf(money).divide(MONEY_MULTIPLIER)

fun toGrutMoney(money: BigDecimal): Long =
    bigDecimalToGrut(money)

fun moneyFromGrut(money: Long): BigDecimal =
    bigDecimalFromGrut(money)

fun moscowDateTimeToGrut(dateTime: LocalDateTime) = DateTimeUtils.moscowDateTimeToEpochSecond(dateTime).toInt()

fun moscowDateTimeFromGrut(epochSeconds: Int): LocalDateTime =
    Instant.ofEpochSecond(epochSeconds.toLong()).atZone(MSK).toLocalDateTime()

// resembles GMT+3 logic from https://a.yandex-team.ru/arc/trunk/arcadia/grut/libs/experimental/parsers/dates_parser.h?rev=r9058291#L13
fun moscowLocalDateToGrut(date: LocalDate) = DateTimeUtils.moscowDateTimeToEpochSecond(date.atTime(OffsetTime.MIN).toLocalDateTime()).toInt()

fun moscowLocalDateFromGrut(epochSeconds: Int) = Instant.ofEpochSecond(epochSeconds.toLong()).atZone(MSK).toLocalDate()

/**
 * Преобразует строковое RGB-представление вида "#AB41CE" в Long (побайтово)
 * Первый символ должен быть равен '#', а длина строки должна быть 7 символов
 */
fun colorStringToUint64(color: String): Long {
    check(color.length == 7)
    check(color[0] == '#')

    val r = Integer.parseInt(color.substring(1, 3), 16)
    val g = Integer.parseInt(color.substring(3, 5), 16)
    val b = Integer.parseInt(color.substring(5, 7), 16)

    return ((r shl 16) or (g shl 8) or b).toLong()
}

/**
 * Преобразует числовое представление цвета в строковое RGB-представление вида "#AB41CE"
 */
fun uint64ToColorString(color: Long): String {
    check(color and 0xffffff == color)

    val lsbMask = 0xffL
    val r = (color shr 16) and lsbMask
    val g = (color shr 8) and lsbMask
    val b = color and lsbMask

    return "#%02X%02X%02X".format(r, g, b)
}

/**
 * Преобразует строковое представление CRC64 в Long (побайтово)
 * Длина строки должна быть 16 символов
 */
fun crc64StringToLong(crc64: String): Long {
    check(crc64.length == 16)
    var longValue: Long = 0
    var i = 0
    while (i < 16) {
        val byte = Integer.parseInt(crc64.substring(i, i + 2), 16)
        longValue = (longValue shl 8) + byte
        i += 2
    }
    return longValue
}
