package ru.yandex.direct.core.grut.converter

import com.google.common.base.Preconditions.checkState
import ru.yandex.direct.core.aggregatedstatuses.BannerDataForStatus
import ru.yandex.direct.core.entity.StatusBsSynced
import ru.yandex.direct.core.entity.banner.aggrstatus.StatusAggregationBanner
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerFlags
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate
import ru.yandex.direct.core.entity.banner.model.old.StatusSitelinksModerate
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag
import ru.yandex.direct.core.entity.moderationreason.model.BannerAssetType
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType
import ru.yandex.direct.core.grut.api.VersionedGrutObject
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType
import ru.yandex.grut.objects.proto.BannerV2
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Status.TModerationStatus.EVerdict
import ru.yandex.grut.objects.proto.client.Schema.TBannerV2

fun convertToBannerDataForStatus(
    bannerCandidate: VersionedGrutObject<TBannerV2>,
    bannerV2: TBannerV2?,
    campaignIdToType: MutableMap<Long, CampaignsType>,
): BannerDataForStatus {
    val campaignType = campaignIdToType[bannerCandidate.grutObject.meta.campaignId]

    val statusAggregationBanner = StatusAggregationBanner()
        .fillMetaFields(bannerCandidate.grutObject, campaignType)
        .fillStatusFields(bannerCandidate, bannerV2)

    return BannerDataForStatus(statusAggregationBanner, emptyMap())
}

private fun StatusAggregationBanner.fillMetaFields(
    bannerCandidate: TBannerV2,
    campaignType: CampaignsType?
): StatusAggregationBanner {
    id = bannerCandidate.meta.directId
    adgroupId = bannerCandidate.meta.adGroupId
    bsBannerId = bannerCandidate.meta.id
    bannerType = bannerCandidate.meta.directType.toBannerType()
    this.campaignType = campaignType
    return this
}

private fun StatusAggregationBanner.fillStatusFields(
    bannerCandidate: VersionedGrutObject<TBannerV2>,
    bannerV2: TBannerV2?
): StatusAggregationBanner {
    statusBsSynced = calcStatusBsSynced(bannerCandidate, bannerV2)
    statusArchived = bannerCandidate.grutObject.spec.status == BannerV2.EBannerStatus.BST_ARCHIVED_VALUE
    statusActive = bannerV2 != null && bannerV2.spec.status == BannerV2.EBannerStatus.BST_ACTIVE_VALUE
    statusShow = bannerCandidate.grutObject.spec.status == BannerV2.EBannerStatus.BST_ACTIVE_VALUE
        || bannerCandidate.grutObject.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE
    statusModerate = calcStatusModerate(bannerCandidate.grutObject)
    statusPostModerate = calcStatusPostModerate(bannerCandidate.grutObject)

    return this
}

private fun calcStatusBsSynced(
    bannerCandidate: VersionedGrutObject<TBannerV2>,
    bannerV2: TBannerV2?
): StatusBsSynced {
    if (bannerCandidate.grutObject.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE) {
        return StatusBsSynced.NO
    }

    if (bannerCandidate.grutObject.spec.status == BannerV2.EBannerStatus.BST_STOPPED_VALUE
        || bannerCandidate.grutObject.spec.status == BannerV2.EBannerStatus.BST_ARCHIVED_VALUE) {
        return if (bannerV2 == null) {
            StatusBsSynced.YES
        } else {
            StatusBsSynced.NO
        }
    }

    return if (bannerV2 == null) {
        StatusBsSynced.NO
    } else if (bannerCandidate.timestamp == bannerV2.status.moderationStatus.mainStatus.version) {
        StatusBsSynced.YES
    } else {
        StatusBsSynced.NO
    }
}

fun calcStatusModerate(
    bannerCandidate: TBannerV2,
): BannerStatusModerate {
    if (bannerCandidate.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE) {
        return BannerStatusModerate.NEW
    }

    if (!bannerCandidate.status.moderationStatus.mainStatus.hasVerdict()) {
        return BannerStatusModerate.SENT
    }

    return bannerCandidate.status.moderationStatus.mainStatus.verdict.toStatusModerate()
}

private fun calcStatusPostModerate(
    bannerCandidate: TBannerV2,
): BannerStatusPostModerate {
    if (bannerCandidate.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE) {
        return BannerStatusPostModerate.NO
    }

    if (!bannerCandidate.status.moderationStatus.mainStatus.hasVerdict()) {
        return BannerStatusPostModerate.NO
    }

    return when (bannerCandidate.status.moderationStatus.mainStatus.verdict.toStatusModerate()) {
        BannerStatusModerate.YES -> BannerStatusPostModerate.YES
        BannerStatusModerate.NO -> BannerStatusPostModerate.REJECTED
        else -> BannerStatusPostModerate.NO
    }
}

private fun Int.toStatusModerate(): BannerStatusModerate {
    return when (this) {
        EVerdict.MV_ACCEPTED_VALUE -> BannerStatusModerate.YES
        EVerdict.MV_REJECTED_VALUE -> BannerStatusModerate.NO
        else -> BannerStatusModerate.SENT
    }
}

private fun Int.toBannerType(): BannersBannerType? {
    return when (this) {
        BannerV2.EBannerType.BT_TEXT_VALUE -> BannersBannerType.text
        BannerV2.EBannerType.BT_DYNAMIC_VALUE -> BannersBannerType.dynamic
        BannerV2.EBannerType.BT_MOBILE_CONTENT_VALUE -> BannersBannerType.mobile_content
        BannerV2.EBannerType.BT_SMART_VALUE -> BannersBannerType.performance
        BannerV2.EBannerType.BT_IMAGE_AD_VALUE -> BannersBannerType.image_ad
        BannerV2.EBannerType.BT_MEDIA_CONTEXT_VALUE -> BannersBannerType.mcbanner
        BannerV2.EBannerType.BT_CPM_VALUE -> BannersBannerType.cpm_banner
        else -> null
    }
}

fun BannerStatusModerate?.toEnumValue(): Int {
    return when (this) {
        BannerStatusModerate.YES -> EVerdict.MV_ACCEPTED_VALUE
        BannerStatusModerate.NO -> EVerdict.MV_REJECTED_VALUE
        else -> EVerdict.MV_UNKNOWN_VALUE
    }
}

fun BannersBannerType?.toEnumValue(): Int {
    return when (this) {
        BannersBannerType.text -> BannerV2.EBannerType.BT_TEXT_VALUE
        BannersBannerType.dynamic -> BannerV2.EBannerType.BT_DYNAMIC_VALUE
        BannersBannerType.mobile_content -> BannerV2.EBannerType.BT_MOBILE_CONTENT_VALUE
        BannersBannerType.performance -> BannerV2.EBannerType.BT_SMART_VALUE
        BannersBannerType.image_ad -> BannerV2.EBannerType.BT_IMAGE_AD_VALUE
        BannersBannerType.mcbanner -> BannerV2.EBannerType.BT_MEDIA_CONTEXT_VALUE
        BannersBannerType.cpm_banner -> BannerV2.EBannerType.BT_CPM_VALUE
        else -> BannerV2.EBannerType.BT_UNKNOWN_VALUE
    }
}

fun calcBannerImageStatusModerate(
    bannerCandidate: TBannerV2,
): StatusBannerImageModerate? {
    if (bannerCandidate.spec.imagesCount == 0) {
        return null
    }

    if (bannerCandidate.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE) {
        return StatusBannerImageModerate.NEW
    }

    if (bannerCandidate.status.moderationStatus.imagesStatusCount == 0) {
        return StatusBannerImageModerate.SENT
    }

    checkState(bannerCandidate.status.moderationStatus.imagesStatusCount == 1)
    return bannerCandidate.status.moderationStatus.getImagesStatus(0).verdict.toBannerImageStatusModerate()
}

private fun Int.toBannerImageStatusModerate(): StatusBannerImageModerate {
    return when (this) {
        EVerdict.MV_ACCEPTED_VALUE -> StatusBannerImageModerate.YES
        EVerdict.MV_REJECTED_VALUE -> StatusBannerImageModerate.NO
        else -> StatusBannerImageModerate.SENT
    }
}

fun calcBannerCreativeStatusModerate(bannerCandidate: TBannerV2): BannerCreativeStatusModerate? {
    if (bannerCandidate.spec.creativeIdsCount == 0) {
        return null
    }

    if (bannerCandidate.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE) {
        return BannerCreativeStatusModerate.NEW
    }

    if (bannerCandidate.status.moderationStatus.creativesStatusCount == 0) {
        return BannerCreativeStatusModerate.SENT
    }

    checkState(bannerCandidate.status.moderationStatus.creativesStatusCount == 1)
    return bannerCandidate.status.moderationStatus.getCreativesStatus(0).verdict.toBannerCreativeStatusModerate()
}

private fun Int.toBannerCreativeStatusModerate(): BannerCreativeStatusModerate {
    return when (this) {
        EVerdict.MV_ACCEPTED_VALUE -> BannerCreativeStatusModerate.YES
        EVerdict.MV_REJECTED_VALUE -> BannerCreativeStatusModerate.NO
        else -> BannerCreativeStatusModerate.SENT
    }
}

fun calcSiteinksSetStatusModerate(bannerCandidate: TBannerV2): StatusSitelinksModerate? {
    if (!bannerCandidate.spec.hasSitelinkSet()) {
        return null
    }

    if (bannerCandidate.spec.status == BannerV2.EBannerStatus.BST_DRAFT_VALUE) {
        return StatusSitelinksModerate.NEW
    }

    if (!bannerCandidate.status.moderationStatus.hasSitelinkSetStatus()) {
        return StatusSitelinksModerate.SENT
    }

    return bannerCandidate.status.moderationStatus.sitelinkSetStatus.verdict.toSiteinksSetStatusModerate()
}

private fun Int.toSiteinksSetStatusModerate(): StatusSitelinksModerate {
    return when (this) {
        EVerdict.MV_ACCEPTED_VALUE -> StatusSitelinksModerate.YES
        EVerdict.MV_REJECTED_VALUE -> StatusSitelinksModerate.NO
        else -> StatusSitelinksModerate.SENT
    }
}

// Not implemented yet
fun calcBannerFlags(bannerCandidate: TBannerV2): BannerFlags? {
    return null
}

// Not implemented yet
fun convertToModerationReasons(banner: TBannerV2): Map<ModerationReasonObjectType, List<ModerationDiag>> {
    return emptyMap()
}

// Not implemented yet
fun convertToAssetModerationReasons(banner: TBannerV2): Map<BannerAssetType, List<ModerationDiag>> {
    return emptyMap()
}
