package ru.yandex.direct.core.mysql2grut.repository

import com.google.common.collect.Multimap
import one.util.streamex.EntryStream
import org.jooq.Record
import org.jooq.Result
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.bidmodifier.BidModifier
import ru.yandex.direct.core.entity.bidmodifier.BidModifierAdjustment
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType
import ru.yandex.direct.core.entity.bidmodifiers.repository.mapper.Common.ALL_BIDMODIFIER_FIELDS
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupport
import ru.yandex.direct.core.entity.bidmodifiers.repository.typesupport.BidModifierTypeSupportDispatcher
import ru.yandex.direct.dbschema.ppc.tables.ExpressionMultiplierValues.EXPRESSION_MULTIPLIER_VALUES
import ru.yandex.direct.dbschema.ppc.tables.HierarchicalMultipliers.HIERARCHICAL_MULTIPLIERS
import ru.yandex.direct.dbschema.ppc.tables.RetargetingMultiplierValues.RETARGETING_MULTIPLIER_VALUES
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import java.util.function.Function

/**
 * Методы работы с корректировками, которые нужны только для репликации и не вписываются в общий api BidModifierRepository
 */
@Repository
class BidModifierReplicationRepository(
    private val dslContextProvider: DslContextProvider,
    val typeSupportDispatcher: BidModifierTypeSupportDispatcher,
) {


    /**
     * В таблице expression_multiplier_values лежат уточнения корректировок для разных типов:
     * traffic_jam и prisma_income_grade
     * В таблице retargeting_multiplier_values лежат уточнения корректировок для двух разных типов:
     * retargeting и retargeting_filter;
     * * Метод отдает записи из родительский таблицы hierarchical_multipliers, содержащие конкретный тип корректировки,
     * без загрузки записией из expression_multiplier_values
     * @param ids -  список идентификаторов из таблицы expression_multiplier_values
     * @param type - какая из таблиц используется для джоина - expression_multiplier_values или retargeting_multiplier_values
     */
    fun getEmptyAmbiguousBidModifiers(shard: Int, type: AmbiguousBidModifierTable, ids: Collection<Long?>?): List<BidModifier> {
        val selectQueryPart = dslContextProvider.ppc(shard)
            .select(ALL_BIDMODIFIER_FIELDS)
            .from(HIERARCHICAL_MULTIPLIERS)

        val fullQuery = when (type) {
            AmbiguousBidModifierTable.RETARGETING_OR_RETARGEING_FILTER ->
                selectQueryPart.join(RETARGETING_MULTIPLIER_VALUES)
                    .on(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.eq(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID))
                    .where(RETARGETING_MULTIPLIER_VALUES.RETARGETING_MULTIPLIER_VALUE_ID.`in`(ids))

            AmbiguousBidModifierTable.EXPRESSION ->
                selectQueryPart.join(EXPRESSION_MULTIPLIER_VALUES)
                    .on(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.eq(EXPRESSION_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID))
                    .where(EXPRESSION_MULTIPLIER_VALUES.EXPRESSION_MULTIPLIER_VALUE_ID.`in`(ids))
        }

        val records: Result<Record> = fullQuery.fetch()
        val recordTypeMultimap: Multimap<BidModifierType, Record> = typeSupportDispatcher.groupRecordsByTypes(records)
        return EntryStream.of(recordTypeMultimap.asMap())
            .mapKeys(Function<BidModifierType, BidModifierTypeSupport<BidModifier, BidModifierAdjustment>> { type: BidModifierType -> typeSupportDispatcher.getTypeSupport(type) })
            .mapKeyValue { obj: BidModifierTypeSupport<BidModifier, BidModifierAdjustment>, records: Collection<Record> -> obj.createEmptyBidModifiersFromRecords(records) }
            .flatMap { bm: List<BidModifier> -> bm.stream() }
            .toList()
    }

    enum class AmbiguousBidModifierTable {
        EXPRESSION,
        RETARGETING_OR_RETARGEING_FILTER,
    }
}
