package ru.yandex.direct.core.mysql2grut.repository

import org.apache.commons.lang.StringUtils.isBlank
import org.jooq.Record
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType
import java.math.BigDecimal
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchMapping
import ru.yandex.direct.core.grut.api.BiddableShowConditionType
import ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS
import ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE
import ru.yandex.direct.dbschema.ppc.tables.BidsDynamic.BIDS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.tables.BidsHrefParams.BIDS_HREF_PARAMS
import ru.yandex.direct.dbschema.ppc.tables.BidsPerformance.BIDS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.tables.DynamicConditions.DYNAMIC_CONDITIONS
import ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES

@Repository
open class BiddableShowConditionsRepository(private val dslContextProvider: DslContextProvider) {
    private val CHUNK_SIZE = 500

    fun getBiddableShowConditions(shard: Int, idsByType: Map<BiddableShowConditionType, Set<Long>>): List<BiddableShowCondition> {
        return idsByType.flatMap { getBiddableShowConditions(shard, it.key, it.value) }
    }

    fun getBiddableShowConditions(shard: Int, type: BiddableShowConditionType, ids: Set<Long>): List<BiddableShowCondition> {
        if (ids.isEmpty()) return emptyList()
        return when (type) {
            BiddableShowConditionType.KEYWORD -> getKeywords(shard, ids)
            BiddableShowConditionType.OFFER_RETARGETING -> getOfferRetargeting(shard, ids)
            BiddableShowConditionType.RELEVANCE_MATCH -> getRelevanceMatch(shard, ids)
            BiddableShowConditionType.DYNAMIC -> getDynamic(shard, ids)
            BiddableShowConditionType.PERFORMANCE -> getPerformance(shard, ids)
            BiddableShowConditionType.RETARGETING -> getRetargeting(shard, ids)
        }
    }

    /**
     * Получить идентификаторы biddable_show_conditions сгруппированные по группам и типу
     */
    fun getAdGroupsBiddableShowConditions(shard: Int, adGroupIds: List<Long>): Map<Long, Map<BiddableShowConditionType, List<Long>>> {
        if (adGroupIds.isEmpty()) return emptyMap()
        val result = mutableMapOf<Long, MutableMap<BiddableShowConditionType, List<Long>>>()

        // keywords
        val keywordIdsByPid = adGroupIds.chunked(CHUNK_SIZE)
            .map { getAdGroupsKeywordsIds(shard, it) }
            .reduce { acc, m -> acc + m }
        for ((pid, bidIds) in keywordIdsByPid) {
            result.getOrPut(pid) { mutableMapOf() }[BiddableShowConditionType.KEYWORD] = bidIds
        }

        // offer retargeting
        val offerRetargetingIdsByPid = adGroupIds.chunked(CHUNK_SIZE)
            .map { getAdGroupsOfferRetargetingIds(shard, it) }
            .reduce { acc, m -> acc + m }
        for ((pid, offerRetargetingIds) in offerRetargetingIdsByPid) {
            result.getOrPut(pid) { mutableMapOf() }[BiddableShowConditionType.OFFER_RETARGETING] = offerRetargetingIds
        }

        // relevance match
        val relevanceMatchIdsByPid = adGroupIds.chunked(CHUNK_SIZE)
            .map { getAdGroupsRelevanceMatchIds(shard, it) }
            .reduce { acc, m -> acc + m }
        for ((pid, relevanceMatchIds) in relevanceMatchIdsByPid) {
            result.getOrPut(pid) { mutableMapOf() }[BiddableShowConditionType.RELEVANCE_MATCH] = relevanceMatchIds
        }

        // dynamic
        val dynamicIdsByPid = adGroupIds.chunked(CHUNK_SIZE)
            .map { getAdGroupsDynamicIds(shard, it) }
            .reduce { acc, m -> acc + m }
        for ((pid, dynamicIds) in dynamicIdsByPid) {
            result.getOrPut(pid) { mutableMapOf() }[BiddableShowConditionType.DYNAMIC] = dynamicIds
        }

        // performance
        val performanceIdsByPid = adGroupIds.chunked(CHUNK_SIZE)
            .map { getAdGroupsPerformanceIds(shard, it) }
            .reduce { acc, m -> acc + m }
        for ((pid, performanceIds) in performanceIdsByPid) {
            result.getOrPut(pid) { mutableMapOf() }[BiddableShowConditionType.PERFORMANCE] = performanceIds
        }

        // retargeting
        val retargetingIdsByPid = adGroupIds.chunked(CHUNK_SIZE)
            .map { getAdGroupsRetargetingBidsIds(shard, it) }
            .reduce { acc, m -> acc + m }
        for((pid, retBidIds) in retargetingIdsByPid) {
            result.getOrPut(pid) { mutableMapOf() }[BiddableShowConditionType.RETARGETING] = retBidIds
        }

        return result
    }

    /**
     * Получить идентификаторы кейвордов, сгруппированных по группам
     */
    fun getAdGroupsKeywordsIds(shard: Int, adGroupIds: List<Long>): Map<Long, List<Long>> {
        val bidsIdsWithPid = dslContextProvider.ppc(shard)
            .select(BIDS.ID, BIDS.PID)
            .from(BIDS)
            .where(BIDS.PID.`in`(adGroupIds))
            .fetch(Companion::mapToBiddableShowConditionIdWithPid)
        return bidsIdsWithPid.groupBy({ it.adGroupId }, { it.id })
    }

    /**
     * Получить идентификаторы retargeting_bids, сгруппированные по группам
     */
    fun getAdGroupsRetargetingBidsIds(shard: Int, adGroupIds: List<Long>): Map<Long, List<Long>> {
        val bidsIdsWithPid = dslContextProvider.ppc(shard)
            .select(BIDS_RETARGETING.RET_ID, BIDS_RETARGETING.PID)
            .from(BIDS_RETARGETING)
            .where(BIDS_RETARGETING.PID.`in`(adGroupIds))
            .fetch(Companion::mapToRetargetingBidWithPid)
        return bidsIdsWithPid.groupBy({ it.adGroupId }, { it.id })
    }


    /**
     * Получить идентификаторы offer_retargeting, сгруппированные по группам
     */
    fun getAdGroupsOfferRetargetingIds(shard: Int, adGroupIds: List<Long>): Map<Long, List<Long>> {
        val bidsBaseIdsWithPid = dslContextProvider.ppc(shard)
            .selectDistinct(BIDS_BASE.BID_ID, BIDS_BASE.PID).from(BIDS_BASE)
            .where(BIDS_BASE.PID.`in`(adGroupIds).and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting)))
            .fetch(Companion::mapToAdGroupBidsBaseId)
        return bidsBaseIdsWithPid.groupBy({ it.adGroupId }, { it.id })
    }


    /**
     * Получить идентификаторы relevance_match, сгруппированные по группам
     */
    fun getAdGroupsRelevanceMatchIds(shard: Int, adGroupIds: List<Long>): Map<Long, List<Long>> {
        val bidsBaseIdsWithPid = dslContextProvider.ppc(shard)
            .selectDistinct(BIDS_BASE.BID_ID, BIDS_BASE.PID).from(BIDS_BASE)
            .where(BIDS_BASE.PID.`in`(adGroupIds).and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match)))
            .fetch(Companion::mapToAdGroupBidsBaseId)
        return bidsBaseIdsWithPid.groupBy({ it.adGroupId }, { it.id })
    }

    /**
     * Получить идентификаторы dynamic_bids, сгруппированные по группам
     */
    fun getAdGroupsDynamicIds(shard: Int, adGroupIds: List<Long>): Map<Long, List<Long>> {
        val dynamicIdsWithPid = dslContextProvider.ppc(shard)
            .selectDistinct(BIDS_DYNAMIC.DYN_ID, BIDS_DYNAMIC.PID).from(BIDS_DYNAMIC)
            .where(BIDS_DYNAMIC.PID.`in`(adGroupIds))
            .fetch(Companion::mapToDynamicIdWithPid)
        return dynamicIdsWithPid.groupBy({ it.adGroupId }, { it.id })
    }

    /**
     * Получить идентификаторы bids_performance, сгруппированные по группам
     */
    fun getAdGroupsPerformanceIds(shard: Int, adGroupIds: List<Long>): Map<Long, List<Long>> {
        val dynamicIdsWithPid = dslContextProvider.ppc(shard)
            .selectDistinct(BIDS_PERFORMANCE.PERF_FILTER_ID, BIDS_PERFORMANCE.PID).from(BIDS_PERFORMANCE)
            .where(BIDS_PERFORMANCE.PID.`in`(adGroupIds))
            .fetch(Companion::mapToPerformanceIdWithPid)
        return dynamicIdsWithPid.groupBy({ it.adGroupId }, { it.id })
    }

    fun getKeywords(shard: Int, ids: Collection<Long>): List<BiddableShowCondition> {
        if (ids.isEmpty()) return emptyList()
        return dslContextProvider.ppc(shard)
            .select(
                BIDS.PID,
                CAMPAIGNS.CLIENT_ID,
                BIDS.ID,
                BIDS.PRICE,
                BIDS.PRICE_CONTEXT,
                BIDS.PHRASE,
                BIDS.SHOWS_FORECAST,
                BIDS.IS_SUSPENDED,
                BIDS_HREF_PARAMS.PARAM1,
                BIDS_HREF_PARAMS.PARAM2,
            ).from(BIDS)
            .join(CAMPAIGNS).on(BIDS.CID.eq(CAMPAIGNS.CID))
            .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.CID.eq(BIDS.CID)
                .and(BIDS_HREF_PARAMS.ID.eq(BIDS.ID)))
            .where(BIDS.ID.`in`(ids))
            .fetch(Companion::mapToKeyword)
    }

    fun getDynamic(shard: Int, ids: Collection<Long>): List<BiddableShowCondition> {
        return dslContextProvider.ppc(shard).select(
            BIDS_DYNAMIC.PID,
            CAMPAIGNS.CLIENT_ID,
            BIDS_DYNAMIC.DYN_ID,
            BIDS_DYNAMIC.DYN_COND_ID,
            BIDS_DYNAMIC.PRICE,
            BIDS_DYNAMIC.PRICE_CONTEXT,
            BIDS_DYNAMIC.OPTS,
            DYNAMIC_CONDITIONS.CONDITION_NAME
        )
            .from(BIDS_DYNAMIC)
            .join(DYNAMIC_CONDITIONS).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
            .join(PHRASES).on(BIDS_DYNAMIC.PID.eq(PHRASES.PID))
            .join(CAMPAIGNS).on(PHRASES.CID.eq(CAMPAIGNS.CID))
            .where(BIDS_DYNAMIC.DYN_ID.`in`(ids))
            .fetch(Companion::mapToDynamicBid)
    }

    fun getPerformance(shard: Int, ids: Collection<Long>): List<BiddableShowCondition> {
        return dslContextProvider.ppc(shard).select(
            BIDS_PERFORMANCE.PID,
            CAMPAIGNS.CLIENT_ID,
            BIDS_PERFORMANCE.PERF_FILTER_ID,
            BIDS_PERFORMANCE.NAME,
            BIDS_PERFORMANCE.PRICE_CPC,
            BIDS_PERFORMANCE.PRICE_CPA,
            BIDS_PERFORMANCE.TARGET_FUNNEL,
            BIDS_PERFORMANCE.IS_SUSPENDED,
            BIDS_PERFORMANCE.IS_DELETED,
        )
            .from(BIDS_PERFORMANCE)
            .join(PHRASES).on(BIDS_PERFORMANCE.PID.eq(PHRASES.PID))
            .join(CAMPAIGNS).on(PHRASES.CID.eq(CAMPAIGNS.CID))
            .where(BIDS_PERFORMANCE.PERF_FILTER_ID.`in`(ids))
            .fetch(Companion::mapToPerformanceBid)
    }

    fun getRetargeting(shard: Int, ids: Collection<Long>): List<BiddableShowCondition> {
        if (ids.isEmpty()) return emptyList()
        return dslContextProvider.ppc(shard).select(
            BIDS_RETARGETING.PID,
            CAMPAIGNS.CLIENT_ID,
            BIDS_RETARGETING.RET_ID,
            BIDS_RETARGETING.RET_COND_ID,
            BIDS_RETARGETING.PRICE_CONTEXT,
        )
            .from(BIDS_RETARGETING)
            .join(CAMPAIGNS).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
            .where(BIDS_RETARGETING.RET_ID.`in`(ids))
            .fetch(Companion::mapToRetargetingBid)
    }

    /**
     * По переданному списку идентификаторов возвращает те идентификаторы, которые существуют в MySql
     */
    fun getExistingKeywords(shard: Int, ids: Collection<Long>): Set<Long> {
        if (ids.isEmpty()) return emptySet()
        return dslContextProvider.ppc(shard)
            .select(
                BIDS.ID
            ).from(BIDS).where(BIDS.ID.`in`(ids))
            .fetch(BIDS.ID)
            .toSet()
    }

    fun getExistingDynamic(shard: Int, ids: Collection<Long>): Set<Long> {
        if (ids.isEmpty()) return emptySet()
        return dslContextProvider.ppc(shard)
            .select(
                BIDS_DYNAMIC.DYN_ID
            ).from(BIDS_DYNAMIC).where(BIDS_DYNAMIC.DYN_ID.`in`(ids))
            .fetch(BIDS_DYNAMIC.DYN_ID)
            .toSet()
    }

    fun getExistingPerformance(shard: Int, ids: Collection<Long>): Set<Long> {
        if (ids.isEmpty()) return emptySet()
        return dslContextProvider.ppc(shard)
            .select(
                BIDS_PERFORMANCE.PERF_FILTER_ID
            ).from(BIDS_PERFORMANCE).where(BIDS_PERFORMANCE.PERF_FILTER_ID.`in`(ids))
            .fetch(BIDS_PERFORMANCE.PERF_FILTER_ID)
            .toSet()
    }

    fun getExistingRetargeting(shard: Int, ids: Collection<Long>): Set<Long> {
        if (ids.isEmpty()) return emptySet()
        return dslContextProvider.ppc(shard)
            .select(
                BIDS_RETARGETING.RET_ID
            ).from(BIDS_RETARGETING).where(BIDS_RETARGETING.RET_ID.`in`(ids))
            .fetch(BIDS_RETARGETING.RET_ID)
            .toSet()
    }

    fun getExistingBidsBase(shard: Int, ids: Collection<Long>): Set<Long> {
        if (ids.isEmpty()) return emptySet()
        return dslContextProvider.ppc(shard)
            .select(
                BIDS_BASE.BID_ID
            ).from(BIDS_BASE).where(BIDS_BASE.BID_ID.`in`(ids))
            .fetch(BIDS_BASE.BID_ID)
            .toSet()

    }

    fun getOfferRetargeting(shard: Int, ids: Collection<Long>): List<BiddableShowCondition> {
        if (ids.isEmpty()) return emptyList()
        return dslContextProvider.ppc(shard).select(
            BIDS_BASE.PID,
            BIDS_BASE.BID_ID,
            CAMPAIGNS.CLIENT_ID,
            BIDS_BASE.OPTS,
        )
            .from(BIDS_BASE)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
            .where(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
            .and(BIDS_BASE.BID_ID.`in`(ids))
            .fetch(Companion::mapToOfferRetargeting)
    }

    fun getRelevanceMatch(shard: Int, ids: Collection<Long>): List<BiddableShowCondition> {
        return dslContextProvider.ppc(shard).select(
            BIDS_BASE.PID,
            CAMPAIGNS.CLIENT_ID,
            BIDS_BASE.BID_ID,
            BIDS_BASE.PRICE,
            BIDS_BASE.PRICE_CONTEXT,
            BIDS_BASE.OPTS,
            BIDS_BASE.RELEVANCE_MATCH_CATEGORIES,
            BIDS_HREF_PARAMS.PARAM1,
            BIDS_HREF_PARAMS.PARAM2,
        )
            .from(BIDS_BASE)
            .join(CAMPAIGNS).on(BIDS_BASE.CID.eq(CAMPAIGNS.CID))
            .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.CID.eq(BIDS_BASE.CID)
                .and(BIDS_HREF_PARAMS.ID.eq(BIDS_BASE.BID_ID)))
            .where(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.relevance_match))
            .and(BIDS_BASE.BID_ID.`in`(ids))
            .fetch(Companion::mapToRelevanceMatchBid)
    }

    fun getBidsDynamicIdsByDynamicConditions(shard: Int, dynamicConditionsIds: Collection<Long>): List<Long> {
        if (dynamicConditionsIds.isEmpty()) return emptyList()
        return dslContextProvider.ppc(shard)
            .select(BIDS_DYNAMIC.DYN_ID)
            .from(BIDS_DYNAMIC).where(BIDS_DYNAMIC.DYN_COND_ID.`in`(dynamicConditionsIds))
            .fetch(BIDS_DYNAMIC.DYN_ID)
    }


    companion object {

        private fun mapToBiddableShowConditionIdWithPid(record: Record): BiddableShowConditionIdWithPid {
            return BiddableShowConditionIdWithPid(record.get(BIDS.ID), record.get(BIDS.PID))
        }

        private fun mapToDynamicIdWithPid(record: Record): BiddableShowConditionIdWithPid {
            return BiddableShowConditionIdWithPid(record.get(BIDS_DYNAMIC.DYN_ID), record.get(BIDS_DYNAMIC.PID))
        }

        private fun mapToPerformanceIdWithPid(record: Record): BiddableShowConditionIdWithPid {
            return BiddableShowConditionIdWithPid(record.get(BIDS_PERFORMANCE.PERF_FILTER_ID), record.get(BIDS_PERFORMANCE.PID))
        }

        private fun mapToAdGroupBidsBaseId(record: Record): BiddableShowConditionIdWithPid {
            return BiddableShowConditionIdWithPid(record.get(BIDS_BASE.BID_ID), record.get(BIDS_BASE.PID))
        }

        private fun mapToRetargetingBidWithPid(record: Record): BiddableShowConditionIdWithPid {
            return BiddableShowConditionIdWithPid(record.get(BIDS_RETARGETING.RET_ID), record.get(BIDS_RETARGETING.PID))
        }

        private fun mapToKeyword(record: Record): BiddableShowCondition {
            val keyword = Keyword(
                record.get(BIDS.PRICE),
                record.get(BIDS.PRICE_CONTEXT),
                record.get(BIDS.PHRASE),
                record.get(BIDS.SHOWS_FORECAST),
                record.get(BIDS.IS_SUSPENDED) == 1L,
                record.get(BIDS_HREF_PARAMS.PARAM1),
                record.get(BIDS_HREF_PARAMS.PARAM2),
            )
            return BiddableShowCondition(
                type = BiddableShowConditionType.KEYWORD,
                id = record.get(BIDS.ID),
                adGroupId = record.get(BIDS.PID),
                clientId = record.get(CAMPAIGNS.CLIENT_ID),
                keyword = keyword,
            )
        }

        private fun mapToOfferRetargeting(record: Record): BiddableShowCondition {
            val offerRetargeting = OfferRetargeting(
                isBidsBaseSuspended(record.get(BIDS_BASE.OPTS)),
            )
            return BiddableShowCondition(
                type = BiddableShowConditionType.OFFER_RETARGETING,
                id = record.get(BIDS_BASE.BID_ID),
                adGroupId = record.get(BIDS_BASE.PID),
                clientId = record.get(CAMPAIGNS.CLIENT_ID),
                offerRetargeting = offerRetargeting,
            )
        }

        private fun mapToRelevanceMatchBid(record: Record): BiddableShowCondition {
            val relevanceMatch = RelevanceMatch(
                isBidsBaseSuspended(record.get(BIDS_BASE.OPTS)),
                record.get(BIDS_BASE.PRICE),
                record.get(BIDS_BASE.PRICE_CONTEXT),
                RelevanceMatchMapping.relevanceMatchCategoriesFromDbFormat(record.get(BIDS_BASE.RELEVANCE_MATCH_CATEGORIES)),
                record.get(BIDS_HREF_PARAMS.PARAM1),
                record.get(BIDS_HREF_PARAMS.PARAM2),
            )
            return BiddableShowCondition(
                type = BiddableShowConditionType.RELEVANCE_MATCH,
                id = record.get(BIDS_BASE.BID_ID),
                adGroupId = record.get(BIDS_BASE.PID),
                clientId = record.get(CAMPAIGNS.CLIENT_ID),
                relevanceMatch = relevanceMatch,
            )
        }

        fun mapToPerformanceBid(record: Record): BiddableShowCondition {
            val targetFunnel = TargetFunnel.fromSource(record.get(BIDS_PERFORMANCE.TARGET_FUNNEL))!!
            val performance = Performance(
                record.get(BIDS_PERFORMANCE.NAME),
                record.get(BIDS_PERFORMANCE.PRICE_CPC),
                record.get(BIDS_PERFORMANCE.PRICE_CPA),
                targetFunnel,
                record.get(BIDS_PERFORMANCE.IS_SUSPENDED) == 1L,
                record.get(BIDS_PERFORMANCE.IS_DELETED) == 1L,
            )
            return BiddableShowCondition(
                BiddableShowConditionType.PERFORMANCE,
                record.get(BIDS_PERFORMANCE.PERF_FILTER_ID),
                adGroupId = record.get(BIDS_PERFORMANCE.PID),
                clientId = record.get(CAMPAIGNS.CLIENT_ID),
                performance = performance,
            )
        }


        fun mapToRetargetingBid(record: Record): BiddableShowCondition {
            val retargeting = Retargeting(
                retargetingConditionId = record.get(BIDS_RETARGETING.RET_COND_ID),
                priceContext = record.get(BIDS_RETARGETING.PRICE_CONTEXT),
            )
            return BiddableShowCondition(
                BiddableShowConditionType.RETARGETING,
                id = record.get(BIDS_RETARGETING.RET_ID),
                adGroupId = record.get(BIDS_RETARGETING.PID),
                clientId = record.get(CAMPAIGNS.CLIENT_ID),
                retargeting = retargeting,
            )
        }

        fun mapToDynamicBid(record: Record): BiddableShowCondition {
            val dynamic = Dynamic(
                record.get(BIDS_DYNAMIC.DYN_COND_ID),
                record.get(BIDS_DYNAMIC.PRICE),
                record.get(BIDS_DYNAMIC.PRICE_CONTEXT),
                isBidsBaseSuspended(record.get(BIDS_DYNAMIC.OPTS)), //константы имеют одинаковое значение
                record.get(DYNAMIC_CONDITIONS.CONDITION_NAME)
            )

            return BiddableShowCondition(
                BiddableShowConditionType.DYNAMIC,
                id = record.get(BIDS_DYNAMIC.DYN_ID),
                adGroupId = record.get(BIDS_DYNAMIC.PID),
                clientId = record.get(CAMPAIGNS.CLIENT_ID),
                dynamic = dynamic,
            )
        }

        private fun isBidsBaseSuspended(opts: String?): Boolean {
            return !isBlank(opts) && opts!!.contains(BidBaseOpt.SUSPENDED.typedValue);
        }
    }

}

data class BiddableShowCondition(
    val type: BiddableShowConditionType,
    val id: Long,
    val clientId: Long,
    val adGroupId: Long,
    val keyword: Keyword? = null,
    val offerRetargeting: OfferRetargeting? = null,
    val relevanceMatch: RelevanceMatch? = null,
    val dynamic: Dynamic? = null,
    val performance: Performance? = null,
    val retargeting: Retargeting? = null,
)

data class Keyword(
    val price: BigDecimal,
    val priceContext: BigDecimal,
    val phrase: String,
    val showsForecast: Long?,
    val isSuspended: Boolean,
    val hrefParam1: String?,
    val hrefParam2: String?,
)

data class OfferRetargeting(
    val isSuspended: Boolean,
)

data class RelevanceMatch(
    val isSuspended: Boolean,
    val price: BigDecimal,
    val priceContext: BigDecimal,
    val categories: Collection<RelevanceMatchCategory>,
    val hrefParam1: String?,
    val hrefParam2: String?,
)

data class Dynamic(
    val dynamic_condition_id: Long,
    val price: BigDecimal,
    val priceContext: BigDecimal,
    val isSuspended: Boolean,
    val conditionName: String?,
)

data class Performance(
    val name: String,
    val priceCPC: BigDecimal,
    val priceCPA: BigDecimal,
    val targetFunnel: TargetFunnel,
    val isSuspended: Boolean,
    val isDeleted: Boolean,
)

data class Retargeting(
    val retargetingConditionId: Long,
    val priceContext: BigDecimal,
)

private data class BiddableShowConditionIdWithPid(val id: Long, val adGroupId: Long)
