package ru.yandex.direct.core.redis;

import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import io.lettuce.core.RedisException;
import io.lettuce.core.cluster.api.sync.RedisAdvancedClusterCommands;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.retry.RetryPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static java.util.Collections.singletonMap;
import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;

/**
 * Storage для работы с Redis.
 */
@Component
@ParametersAreNonnullByDefault
public class LettuceStorage {

    private static final Logger logger = LoggerFactory.getLogger(LettuceStorage.class);

    private final LettuceConnectionProvider connectionProvider;
    private final RetryTemplate retryTemplate;

    @Autowired
    public LettuceStorage(@Qualifier(LETTUCE) LettuceConnectionProvider connectionProvider) {
        this.connectionProvider = connectionProvider;

        RetryPolicy policy = new SimpleRetryPolicy(connectionProvider.getMaxAttempts(),
                singletonMap(RedisException.class, true));

        retryTemplate = new RetryTemplate();
        retryTemplate.setRetryPolicy(policy);
    }

    public String get(String key) {
        String value = redisProfiledCall("redis:get", cmd -> cmd.get(key));
        logger.info("get({}): {}", key, value);
        return value;
    }

    public boolean setAndExpire(String key, String value, long ttl) {
        logger.info("setex({}, {}, {})", key, value, ttl);
        return "OK".equals(redisProfiledCall("redis:setex", cmd -> cmd.setex(key, ttl, value)));
    }

    public boolean setAndExpire(String key, Integer value, long ttl) {
        return setAndExpire(key, value.toString(), ttl);
    }

    public String getAndSet(String key, String value) {
        String oldValue = redisProfiledCall("redis:getset", cmd -> cmd.getset(key, value));
        logger.info("getset({}, {}): {}", key, value, oldValue);
        return oldValue;
    }

    public boolean expire(String key, long ttl) {
        logger.info("expire({}, {})", key, ttl);
        return redisProfiledCall("redis:expire", cmd -> cmd.expire(key, ttl));
    }

    public long incrementAndGet(String key) {
        long newValue = redisProfiledCall("redis:incr", cmd -> cmd.incr(key));
        logger.info("incr({}): {}", key, newValue);
        return newValue;
    }

    public long delete(String key) {
        logger.info("del({})", key);
        return redisProfiledCall("redis:del", cmd -> cmd.del(key));
    }

    public long getTtl(String key) {
        long ttl = redisProfiledCall("redis:ttl", cmd -> cmd.ttl(key));
        logger.info("ttl({}): {}", key, ttl);
        return ttl;
    }

    /**
     * Connect to redis and execute function with profiling.
     *
     * @param name     Profiled method name
     * @param function Function to execute on call
     * @param <T>      Return type
     * @return profiled function result
     */
    private <T> T redisProfiledCall(String name, Function<RedisAdvancedClusterCommands<String, String>, T> function) {
        try (TraceProfile ignored = Trace.current().profile(name)) {
            return retryTemplate.execute(context -> function.apply(connectionProvider.getConnection().sync()));
        } catch (Exception e) {
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            throw new StorageErrorException(e);
        }
    }
}
