package ru.yandex.direct.core.security;

import java.net.InetAddress;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import javax.annotation.Nullable;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;

import ru.yandex.direct.core.entity.user.model.User;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

/**
 * Internal final authentication object.
 * Useful for core-level authorization.
 */
public class DirectAuthentication implements Authentication {
    private final User operator;
    private final User subjectUser;
    private final List<SimpleGrantedAuthority> authorities;
    private final String tvmUserTicket;
    private final InetAddress userIp;

    public DirectAuthentication(User operator, User subjectUser) {
        this(operator, subjectUser, null, null);
    }

    public DirectAuthentication(User operator, User subjectUser, @Nullable String tvmUserTicket,
                                @Nullable InetAddress userIp) {
        this.operator = operator;
        this.subjectUser = subjectUser;
        authorities = Collections.singletonList(
                new SimpleGrantedAuthority("ROLE_" + operator.getRole().name()));
        this.tvmUserTicket = tvmUserTicket;
        this.userIp = userIp;
    }

    public User getOperator() {
        return operator;
    }

    public User getSubjectUser() {
        return subjectUser;
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        return authorities;
    }

    /**
     * @return null
     */
    @Override
    @Deprecated
    public final String getPrincipal() {
        return null;
    }

    /**
     * @return operator login
     */
    @Override
    public final String getName() {
        return ifNotNull(getOperator(), User::getLogin);
    }

    /**
     * There is no details for internal Direct authentication
     *
     * @return {@code null}
     */
    @Override
    @Deprecated
    public final Object getDetails() {
        return null;
    }

    /**
     * Internal Direct authentication does not contain credentials
     *
     * @return null
     */
    public final Object getCredentials() {
        return null;
    }

    /**
     * Internal Direct authentication is final
     * and than is always "authenticated".
     *
     * @return {@code true}
     */
    @Override
    @Deprecated
    public final boolean isAuthenticated() {
        return true;
    }

    /**
     * Internal Direct authentication is final
     * and than is always "authenticated".
     */
    @Override
    @Deprecated
    public final void setAuthenticated(boolean isAuthenticated) {
        if (isAuthenticated) {
            throw new IllegalArgumentException("can not set \"authenticated\" property");
        }
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public Optional<String> getTvmUserTicket() {
        return Optional.ofNullable(tvmUserTicket);
    }

    public InetAddress getUserIp() {
        return userIp;
    }
}
