package ru.yandex.direct.core.security.authorization;

import java.io.Serializable;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.rbac.ClientPerm;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.ClientsRelationType;

@Component
public class DirectPermissionEvaluator implements PermissionEvaluator {
    private static final Logger LOGGER = LoggerFactory.getLogger(DirectPermissionEvaluator.class);

    private RbacService rbacService;
    private ClientService clientService;
    private RbacClientsRelations rbacClientsRelations;

    @Autowired
    public DirectPermissionEvaluator(RbacService rbacService, ClientService clientService,
                                     RbacClientsRelations rbacClientsRelations) {
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.rbacClientsRelations = rbacClientsRelations;
    }

    @Override
    public boolean hasPermission(Authentication authentication, Object targetDomainObject, Object permission) {
        return false;
    }

    /**
     * Проверяет права на выполнение операции.
     *
     * @param auth          объект аутентификации (оператор выполняет действия над клиентом)
     * @param targetId      id защищенного объекта, например кампании, к которой проверяется доступ,
     *                      или null, если объект создается или id не важен
     * @param targetTypeStr тип защищенного объекта (не используется)
     * @param permissionObj тип требуемого разрешения, например, чтение или запись
     * @return true, если доступ разрешен, в противном случае - false.
     */
    @Override
    public boolean hasPermission(Authentication auth, Serializable targetId,
                                 String targetTypeStr, Object permissionObj) {
        Permission permission = Permission.smartValueOf(permissionObj);
        DirectAuthentication authentication = (DirectAuthentication) auth;

        return hasPermission(authentication, permission);
    }

    /**
     * Проверка прав Operator -> Client
     */
    public boolean hasPermission(DirectAuthentication authentication, Permission permission) {
        User operator = authentication.getOperator();
        User client = authentication.getSubjectUser();

        // todo ATTENTION: реализован не полностью, при использовании внимательно смотреть, реализован ли ваш кейс
        RbacRole operatorRole = operator.getRole();
        switch (operatorRole) {
            case SUPER:
                return true;
            case SUPERREADER:
                return permission.isRead() || haveSameClientId(operator, client);
            case SUPPORT:
                return true;
            case PLACER:
                return permission.isRead() || haveSameClientId(operator, client);
            case MEDIA:
                return permission.isRead() || haveSameClientId(operator, client);
            case MANAGER:
                return permission.isRead() || haveSameClientId(operator, client)
                        || rbacService.isOwner(operator.getUid(), client.getUid());
            case AGENCY:
                return haveSameClientId(operator, client) || agencyHasPermission(operator, client, permission);
            case INTERNAL_AD_ADMIN:
                return haveSameClientId(operator, client) || client.getPerms().contains(ClientPerm.INTERNAL_AD_PRODUCT);
            case INTERNAL_AD_MANAGER:
                return haveSameClientId(operator, client)
                        || rbacClientsRelations
                        .getInternalAdProductRelation(operator.getClientId(), client.getClientId())
                        .map(relation -> permission.isRead() ||
                                relation.getRelationType() == ClientsRelationType.INTERNAL_AD_PUBLISHER)
                        .orElse(false);
            case INTERNAL_AD_SUPERREADER:
                return haveSameClientId(operator, client) ||
                        (permission.isRead() && client.getPerms().contains(ClientPerm.INTERNAL_AD_PRODUCT));
            case LIMITED_SUPPORT:
                return haveSameClientId(operator, client) ||
                        (permission.isRead() && rbacService.isOwner(operator.getUid(), client.getUid()));
            case CLIENT:
                return clientHasPermission(operator, client, permission);
            default:
                LOGGER.error("unsupported role: {}", operatorRole);
                return false;
        }
    }

    private boolean agencyHasPermission(User operator, User client, Permission permission) {
        if (client.getRole() == RbacRole.AGENCY) {
            // пока права только чтение
            return permission.isRead() && Objects.equals(operator.getClientId().asLong(),
                    client.getClientId().asLong());
        } else {
            return rbacService.canAgencyCreateCampaign(operator.getUid(), client.getUid());
        }
    }

    private boolean clientHasPermission(User operator, User client, Permission permission) {
        if (!haveSameClientId(operator, client)) {
            long effectiveUid = rbacService.getEffectiveOperatorUid(operator.getUid(), client.getUid());

            if (effectiveUid != client.getUid()) {
                return false;
            }
        }
        if (client.getRole() != RbacRole.CLIENT) {
            return false;
        }
        if (permission.isRead()) {
            return true;
        } else if (operator.getIsReadonlyRep()) {
            return false;
        }
        if (permission.isWriteSettings()) {
            return true;
        }
        if (isUnderAgency(client)) {
            return isSuperSubclient(client);
        } else {
            return true;
        }
    }

    private boolean haveSameClientId(User a, User b) {
        return Objects.equals(a.getClientId(), b.getClientId());
    }

    private boolean isUnderAgency(User client) {
        return rbacService.isUnderAgency(client.getUid());
    }

    private boolean isSuperSubclient(User client) {
        return clientService.isSuperSubclient(client.getClientId());
    }
}
