package ru.yandex.direct.core.service.appinfo

import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.uac.model.AppInfo
import ru.yandex.direct.core.entity.uac.model.AppInfoContent
import ru.yandex.direct.core.entity.uac.model.MobileAppStoreUrlWithOriginal
import ru.yandex.direct.core.entity.uac.model.Store
import ru.yandex.direct.core.entity.uac.repository.ydb.UacYdbAppInfoRepository
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbAppInfo
import ru.yandex.direct.core.entity.uac.service.appinfo.GooglePlayAppInfoGetter
import ru.yandex.direct.core.entity.uac.service.appinfo.GooglePlayContentProcessor
import ru.yandex.direct.core.entity.uac.service.appinfo.ITunesAppInfoGetter
import ru.yandex.direct.core.entity.uac.service.appinfo.ItunesContentProcessor
import ru.yandex.direct.geobasehelper.GeoBaseHelper
import java.time.Duration
import java.time.LocalDateTime

@Lazy
@Service
class UacAppInfoParsingService(
    private val googlePlayAppInfoProcessor: GooglePlayContentProcessor,
    private val itunesAppInfoProcessor: ItunesContentProcessor,
    private val appInfoRepository: UacYdbAppInfoRepository,
    private val googlePlayAppInfoGetter: GooglePlayAppInfoGetter,
    private val iTunesAppInfoGetter: ITunesAppInfoGetter,
    private val geoBaseHelper: GeoBaseHelper,
) {

    companion object {
        val DEFAULT_TARGET_REGIONS: Map<String, Long> = mapOf(
            "ua" to 187,
            "by" to 149,
            "ru" to 225,
            "kz" to 159,
            "uk" to 102,
            "us" to 84,
            "de" to 96,
            "tr" to 983,
            "fr" to 124,
            "tm" to 170,
            "uz" to 171,
            "tj" to 209,
            "ge" to 169,
            "am" to 168,
            "az" to 167,
        )
    }

    fun getDefaultTargetRegionWithName(region: String, lang: String): Pair<Long, String>? {
        val regionId = DEFAULT_TARGET_REGIONS[region.lowercase()]
        return if (regionId != null) {
            regionId to geoBaseHelper.getRegionName(regionId, lang)
        } else {
            null
        }
    }

    fun getFromCacheOrParse(urls: MobileAppStoreUrlWithOriginal, cacheTtl: Int): Pair<AppInfo, List<AppInfoContent>> {
        val model = getModelFromCacheOrParse(urls, cacheTtl)
        val getter = when (urls.parsedUrl.store) {
            Store.GOOGLE_PLAY -> googlePlayAppInfoGetter
            Store.ITUNES -> iTunesAppInfoGetter
            else -> throw IllegalArgumentException("Unknown store ${urls.parsedUrl.store} for ${urls.originalUrl}")
        }
        return getter.getAppInfo(model, true) to getter.getContent(model)
    }

    private fun getModelFromCacheOrParse(urls: MobileAppStoreUrlWithOriginal, cacheTtl: Int): UacYdbAppInfo {
        fun canUseCached(cacheTtl: Int) = cacheTtl != 0

        fun infCache(cacheTtl: Int) = cacheTtl == -1

        fun isCacheOutdated(model: UacYdbAppInfo, cacheTtl: Int) =
            !infCache(cacheTtl) && model.updatedAt + Duration.ofSeconds(cacheTtl.toLong()) < LocalDateTime.now()

        return if (canUseCached(cacheTtl)) {
            val cachedModel = appInfoRepository.getAppInfoByUrl(urls.parsedUrl)
            if (cachedModel == null || isCacheOutdated(cachedModel, cacheTtl)) {
                parseAndCache(urls)
            } else {
                cachedModel
            }
        } else {
            parseAndCache(urls)
        }
    }

    private fun parseAndCache(urls: MobileAppStoreUrlWithOriginal): UacYdbAppInfo {
        val parsedModel = when (urls.parsedUrl.store) {
            Store.GOOGLE_PLAY -> googlePlayAppInfoProcessor.getAppInfoModel(urls)
            Store.ITUNES -> itunesAppInfoProcessor.getAppInfoModel(urls)
            else -> throw IllegalArgumentException("Unknown store ${urls.parsedUrl.store} for ${urls.originalUrl}")
        }
        appInfoRepository.saveAppInfo(parsedModel)
        return parsedModel
    }
}

