package ru.yandex.direct.core.service.integration.balance;

import java.math.BigDecimal;
import java.net.InetAddress;
import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.balance.client.exception.BalanceClientException;
import ru.yandex.direct.balance.client.model.request.AgencySelectPolicy;
import ru.yandex.direct.balance.client.model.request.CheckBindingRequest;
import ru.yandex.direct.balance.client.model.request.CreateClientRequest;
import ru.yandex.direct.balance.client.model.request.CreateInvoiceRequest;
import ru.yandex.direct.balance.client.model.request.CreateOrUpdateOrdersBatchRequest;
import ru.yandex.direct.balance.client.model.request.CreatePersonRequest;
import ru.yandex.direct.balance.client.model.request.CreateRequest2Item;
import ru.yandex.direct.balance.client.model.request.FindClientRequest;
import ru.yandex.direct.balance.client.model.request.GetFirmCountryCurrencyRequest;
import ru.yandex.direct.balance.client.model.request.ListPaymentMethodsSimpleRequest;
import ru.yandex.direct.balance.client.model.request.PayRequestRequest;
import ru.yandex.direct.balance.client.model.request.createtransfermultiple.CreateTransferMultipleRequest;
import ru.yandex.direct.balance.client.model.response.CheckBindingResponse;
import ru.yandex.direct.balance.client.model.response.ClientPassportInfo;
import ru.yandex.direct.balance.client.model.response.CreateRequest2Response;
import ru.yandex.direct.balance.client.model.response.FindClientResponseItem;
import ru.yandex.direct.balance.client.model.response.FindClientResult;
import ru.yandex.direct.balance.client.model.response.GetCardBindingURLResponse;
import ru.yandex.direct.balance.client.model.response.GetClientPersonsResponseItem;
import ru.yandex.direct.balance.client.model.response.GetOverdraftParamsResponse;
import ru.yandex.direct.balance.client.model.response.GetRequestChoicesResponse;
import ru.yandex.direct.balance.client.model.response.GetRequestChoicesResponseFull;
import ru.yandex.direct.balance.client.model.response.ListPaymentMethodsSimpleResponseItem;
import ru.yandex.direct.balance.client.model.response.PayRequestResponse;
import ru.yandex.direct.balance.client.model.response.PaymentMethodDetails;
import ru.yandex.direct.balance.client.model.response.PaymentMethodDetailsFull;
import ru.yandex.direct.balance.client.model.response.Person;
import ru.yandex.direct.balance.client.model.response.PersonPaymentMethodDetails;
import ru.yandex.direct.balance.client.model.response.PersonPaymentMethodDetailsFull;
import ru.yandex.direct.core.entity.balance.container.PaymentMethodInfo;
import ru.yandex.direct.core.entity.balance.container.PersonInfo;
import ru.yandex.direct.core.entity.balance.container.PersonPaymentMethodInfo;
import ru.yandex.direct.core.entity.payment.model.CardInfo;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.service.integration.balance.defects.BalanceDefectIds;
import ru.yandex.direct.core.service.integration.balance.defects.BalanceNumericDefectIds;
import ru.yandex.direct.core.service.integration.balance.model.CreateAndPayRequestResult;
import ru.yandex.direct.core.service.integration.balance.model.CreateInvoiceRequestResult;
import ru.yandex.direct.core.service.integration.balance.model.PaymentMethodType;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.balance.client.model.response.GetClientPersonsResponseItem.LEGAL_PERSON_TYPE;
import static ru.yandex.direct.balance.client.model.response.GetClientPersonsResponseItem.NATURAL_PERSON_TYPE;
import static ru.yandex.direct.common.util.GuavaCollectors.toMultimap;
import static ru.yandex.direct.common.util.HttpUtil.getRemoteAddress;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Интеграция с балансом
 */
@ParametersAreNonnullByDefault
public class BalanceService {
    public static final LocalDate BEGIN_OF_TIME_FOR_MULTICURRENCY_CLIENT = LocalDate.of(2000, 1, 1);
    private static final BigDecimal GET_REQUEST_CHOICES_MIN_PAYMENT_SUM = BigDecimal.TEN;
    private static final String LOCALHOST = "127.0.0.1";
    /**
     * Ключ под которым в хэш-таблице Баланс возвращает тип метода платежа.
     * В документации баланса отсутствует, найден опытным путём.
     */
    private static final String PAYMENT_METHODS_TYPE_KEY = "type";
    private static final String DEFAULT_PHONE = "+0";
    private static final String DEFAULT_FIRST_NAME = "-";
    private static final String DEFAULT_LAST_NAME = "Фамилия";
    private static final Pattern PROMOCODE_MIN_VALUE_PATTERN = Pattern.compile("minimal_money (\\d+)");

    private static final Logger logger = LoggerFactory.getLogger(BalanceService.class);

    private final BalanceClient balanceClient;
    private final Integer directServiceId;
    private final String directServiceToken;

    public BalanceService(BalanceClient balanceClient, int directServiceId, String directServiceToken) {
        this.balanceClient = Objects.requireNonNull(balanceClient, "balanceClient");
        this.directServiceId = directServiceId;
        this.directServiceToken = directServiceToken;
    }

    /**
     * Создать клиента в балансе
     * <p>
     * См. direct-utils::Yandex::Balance.pm + Clients.pm
     *
     * @param operatorUid Uid инициатора
     * @param agency      Агенство, используется только для создания клиентов агенства
     * @param newClient   Свойства нового клиента
     * @param isAgency    Создается агенство?
     * @return Идентификатор нового клиента в балансе
     */
    @Nullable
    private ClientId createClient(
            Long operatorUid,
            @Nullable UidAndClientId agency,
            RegisterClientRequest newClient,
            boolean isAgency) {
        try {
            CreateClientRequest request = new CreateClientRequest()
                    .withClientId(newClient.getClientId())
                    .withClientTypeId(nvl(newClient.getClientTypeId(), 0))
                    .withName(newClient.getName())
                    .withEmail(newClient.getEmail())
                    .withPhone("-")
                    .withFax("-")
                    .withUrl("-")
                    .withCity("-")
                    .withIsAgency(isAgency)
                    .withRegionId(ifNotNull(newClient.getCountryRegionId(), Long::intValue))
                    // передаём всегда директовский ServiceID, т.к. клиенты по валюте общие во всех сервисах, живущих
                    // у нас
                    .withServiceId(directServiceId);

            if (agency != null) {
                request.setAgencyId(agency.getClientId().asLong());
            }

            if (CurrencyCode.YND_FIXED != newClient.getCurrency()) {
                request.setCurrency(newClient.getCurrency().name());
                // для новых клиентов отправляем давно прошедшую дату перехода. она нужна Балансу,
                // чтобы понять по новой (мультивалютной) или старой (в фишках) схеме работает клиент
                request.setMigrateToCurrency(BEGIN_OF_TIME_FOR_MULTICURRENCY_CLIENT);
            }

            return ClientId.fromLong(balanceClient.createClient(operatorUid, request));
        } catch (BalanceClientException e) {
            logger.error(
                    String.format(
                            "createClient failed (operatorUid: %d, agencyId: %s, newClientUid: %d)",
                            operatorUid,
                            agency != null ? agency.getClientId().toString() : "null",
                            newClient.getClientUid()),
                    e);
            return null;
        }
    }

    /**
     * Создать представителя клиента (См. direct-utils::Yandex::Balance::create_client_id_association)
     *
     * @return true в случае успеха, иначе false
     */
    private boolean createClientAssociation(Long operatorUid, UidAndClientId newClient) {
        try {
            balanceClient
                    .createUserClientAssociation(operatorUid, newClient.getClientId().asLong(), newClient.getUid());

            // Если у клиента представитель один, то передаем в баланс признак "Главный"
            if (balanceClient.getClientRepresentativePassports(operatorUid, newClient.getClientId().asLong()).size()
                    == 1) {
                balanceClient.editPassport(
                        operatorUid, newClient.getUid(), new ClientPassportInfo().withIsMain(1));
            }

            return true;
        } catch (BalanceClientException e) {
            logger.error(
                    "createClientAssociation failed (operatorUid: {}, newClient: {})",
                    operatorUid, newClient.getClientId(), e);
            return false;
        }
    }

    /**
     * Связывает в балансе новых пользователей с их клиентом.
     * Предполагается, что пользователи уже провалидированы на существование и отсутствие связей с другими клиентами,
     * в противном случае метод может выбросить исключение.
     */
    public void createUserClientAssociations(List<User> newUsers) {
        for (User user : newUsers) {
            balanceClient.createUserClientAssociation(user.getChiefUid(), user.getClientId().asLong(), user.getUid());
        }
    }

    /**
     * Делает пользователя новым главным представителем клиента.
     * Ожидается, что пользователь уже провалидирован на то, что он существует в балансе и связан с заданным клиентом,
     * в противном случае метод может выбросить исключение IllegalStateException.
     */
    public void changeChief(ClientId clientId, Long newChiefUid) {
        long clientAsLong = clientId.asLong();
        List<ClientPassportInfo> representatives =
                balanceClient.getClientRepresentativePassports(clientAsLong, clientAsLong);
        ClientPassportInfo newChief = StreamEx.of(representatives)
                .findAny(representative -> Objects.equals(representative.getUid(), newChiefUid))
                .orElse(null);
        checkState(newChief != null && newChief.getClientId() == clientAsLong);
        // Сначала выставляем isMain=1 для нового представителя, потом сбрасываем для старых.
        updateIsMain(newChiefUid, newChief, 1);
        for (var representative : representatives) {
            if (notEquals(representative.getUid(), newChiefUid)) {
                updateIsMain(newChiefUid, representative, 0);
            }
        }
    }

    private void updateIsMain(Long operatorUid, ClientPassportInfo passportUser, Integer newValue) {
        if (notEquals(passportUser.getIsMain(), newValue)) {
            ClientPassportInfo updateInfo = new ClientPassportInfo().withIsMain(newValue);
            balanceClient.editPassport(operatorUid, passportUser.getUid(), updateInfo);
        }
    }


    /**
     * Отвязывает в Балансе представителя от клиента.
     * Если представитель отсутсвует в Балансе, или привязан там к другому клиенту - метод выбросит исключение
     */
    public void removeUserClientAssociation(User user) {
        balanceClient.removeUserClientAssociation(user.getChiefUid(), user.getClientId().asLong(), user.getUid());
    }

    /**
     * Вызывает метод в Балансе, которым ограниченному представителю агентства назначаются клиенты
     *
     * @param operatorUid  UID оператора
     * @param agencyRep    представитель агентства
     * @param subclientIds список ClientID субклиентов
     */
    public void setAgencyLimitedRepSubclients(Long operatorUid, UidAndClientId agencyRep, List<Long> subclientIds) {
        try {
            balanceClient.setAgencyLimitedRepSubclients(
                    operatorUid, agencyRep.getClientId().asLong(), agencyRep.getUid(), subclientIds);
        } catch (BalanceClientException e) {
            logger.error(
                    "setAgencyLimitedRepSubclients failed (operatorUid: {}, newClient: {})",
                    operatorUid, agencyRep.getClientId(), e);
        }
    }

    /**
     * Получить возможные способы оплаты для указанного ОС на указанную сумму
     * с указанием выбранных ранее плательщика/способа оплаты для выставления автоовердрафтных счетов
     *
     * @param operatorUid uid оператора
     * @param clientId    ID клиента
     * @param walletCid   ID общего счета
     */
    public List<PersonPaymentMethodInfo> getPaymentOptions(Long operatorUid, ClientId clientId, Long walletCid) {
        try {
            CreateRequest2Item orderItem =
                    new CreateRequest2Item()
                            .withQty(GET_REQUEST_CHOICES_MIN_PAYMENT_SUM)
                            .withServiceId(directServiceId)
                            .withServiceOrderId(walletCid);

            GetOverdraftParamsResponse overdraftParams =
                    balanceClient.getOverdraftParams(directServiceId, clientId.asLong());
            GetRequestChoicesResponse paymentMethods =
                    balanceClient.getPaymentMethods(operatorUid, clientId.asLong(), singletonList(orderItem));

            List<PersonPaymentMethodDetails> personPaymentMethodDetails = Optional.ofNullable(paymentMethods)
                    .map(GetRequestChoicesResponse::getPersonPaymentMethodDetails)
                    .orElse(emptyList());

            return filterAndMapList(personPaymentMethodDetails, this::isPersonPaymentMethodDetailsFullyFilled,
                    ppm -> toPersonPaymentMethodInfo(overdraftParams, ppm));
        } catch (BalanceClientException e) {
            logger.error("getPaymentOptions failed (operatorUid: {}, newClient: {})", operatorUid, clientId, e);
            throw e;
        }
    }

    /**
     * Получить возможные способы оплаты для указанного ОС на указанную сумму
     *
     * @param operatorUid   uid оператора
     * @param clientId      ID клиента
     * @param walletCid     ID общего счета
     * @param personIdToPay ID плательщика из Баланса
     */
    public List<PaymentMethodDetailsFull> getPaymentOptionsFull(Long operatorUid, ClientId clientId, Long walletCid,
                                                                Long personIdToPay) {
        try {
            CreateRequest2Item orderItem =
                    new CreateRequest2Item()
                            .withQty(GET_REQUEST_CHOICES_MIN_PAYMENT_SUM)
                            .withServiceId(directServiceId)
                            .withServiceOrderId(walletCid);

            GetRequestChoicesResponseFull paymentMethods =
                    balanceClient.getPaymentMethodsFull(operatorUid, clientId.asLong(), singletonList(orderItem));

            List<PersonPaymentMethodDetailsFull> personPaymentMethodDetails = Optional.ofNullable(paymentMethods)
                    .map(GetRequestChoicesResponseFull::getPersonPaymentMethodDetailsFull)
                    .orElse(emptyList());

            List<PaymentMethodDetailsFull> paymentMethodDetails = personPaymentMethodDetails.stream()
                    .filter(ppm -> isPersonPaymentMethodDetailsFullyFilled(ppm)
                            && filterPersonPaymentMethodInfo(ppm, personIdToPay))
                    .map(PersonPaymentMethodDetailsFull::getPaymentMethodDetailsFull)
                    .findFirst()
                    .orElse(List.of());

            return filterList(paymentMethodDetails, this::isPaymentMethodDetailsFullyFilled);
        } catch (BalanceClientException e) {
            logger.error("getRequestChoices failed (operatorUid: {}, newClient: {})", operatorUid, clientId, e);
            throw e;
        }
    }

    private boolean isPersonPaymentMethodDetailsFullyFilled(PersonPaymentMethodDetails personPaymentMethodDetails) {
        return personPaymentMethodDetails.getPerson() != null
                && personPaymentMethodDetails.getPerson().getId() != null
                && personPaymentMethodDetails.getPerson().getName() != null
                && personPaymentMethodDetails.getPaymentMethodDetails() != null
                && personPaymentMethodDetails.getContract() == null;
    }

    private boolean isPersonPaymentMethodDetailsFullyFilled(PersonPaymentMethodDetailsFull personPaymentMethodDetails) {
        return personPaymentMethodDetails.getPerson() != null
                && personPaymentMethodDetails.getPerson().getId() != null
                && personPaymentMethodDetails.getPerson().getName() != null
                && personPaymentMethodDetails.getPaymentMethodDetailsFull() != null;
    }

    private static boolean filterPersonPaymentMethodInfo(PersonPaymentMethodDetailsFull personPaymentMethodDetailsFull,
                                                         Long personIdToPay) {
        Long personId = personPaymentMethodDetailsFull.getPerson().getId().longValue();
        return personId.equals(personIdToPay);
    }

    private PersonPaymentMethodInfo toPersonPaymentMethodInfo(GetOverdraftParamsResponse overdraftParams,
                                                              PersonPaymentMethodDetails paymentMethodDetails) {
        Person person = paymentMethodDetails.getPerson();
        boolean personIsSelected =
                overdraftParams != null && person.getId().equals(overdraftParams.getPersonId());
        PersonInfo personInfo = new PersonInfo()
                .withId(person.getId().longValue())
                .withName(person.getName())
                .withSelected(personIsSelected);
        List<PaymentMethodInfo> paymentMethodInfoList = StreamEx.of(paymentMethodDetails.getPaymentMethodDetails())
                .filter(this::isPaymentMethodDetailsFullyFilled)
                .distinct()
                .map(pm -> toPaymentMethodInfo(overdraftParams, personIsSelected, pm))
                .filter(pmi -> pmi.getCode() != null && !pmi.getCode().contains("paypal"))
                .toList();
        return new PersonPaymentMethodInfo()
                .withPersonInfo(personInfo)
                .withPaymentMethods(paymentMethodInfoList);
    }

    private boolean isPaymentMethodDetailsFullyFilled(PaymentMethodDetails paymentMethodDetails) {
        return paymentMethodDetails.getPaymentMethod() != null;
    }

    private boolean isPaymentMethodDetailsFullyFilled(PaymentMethodDetailsFull paymentMethodDetails) {
        return paymentMethodDetails.getId() != null
                && paymentMethodDetails.getName() != null
                && paymentMethodDetails.getPaymentMethod() != null;
    }

    private PaymentMethodInfo toPaymentMethodInfo(GetOverdraftParamsResponse overdraftParams, boolean personIsSelected,
                                                  PaymentMethodDetails pm) {
        String code = pm.getPaymentMethod().getCode();
        String name = pm.getPaymentMethod().getName();
        BigDecimal limit = pm.getPaymentLimit();
        return new PaymentMethodInfo()
                .withName(name)
                .withCode(code)
                .withLimit(limit)
                .withSelected(personIsSelected && code.equals(overdraftParams.getPaymentMethodCode()));
    }

    public void setOverdraftParams(Long personId, String paymentMethodCode, String isoCurrencyCode,
                                   BigDecimal clientLimit) {
        balanceClient.setOverdraftParams(personId, directServiceId, paymentMethodCode, isoCurrencyCode, clientLimit);
    }

    public void resetOverdraftParams(ClientId clientId, String isoCurrencyCode) {
        GetOverdraftParamsResponse overdraftParams =
                balanceClient.getOverdraftParams(directServiceId, clientId.asLong());
        boolean clientHasAutoOverdraftLimit = overdraftParams != null
                && overdraftParams.getClientLimit() != null
                && overdraftParams.getClientLimit().compareTo(BigDecimal.ZERO) > 0;
        if (clientHasAutoOverdraftLimit) {
            balanceClient.setOverdraftParams(overdraftParams.getPersonId().longValue(), directServiceId,
                    overdraftParams.getPaymentMethodCode(),
                    isoCurrencyCode, BigDecimal.ZERO);
        }
    }

    /**
     * Зарегистрировать нового клиента в балансе.
     * <p>
     * См. Clients.pm::CreateNewSubclient
     *
     * @param operatorUid UID оператора
     * @param agency      Агентство для субклиента. Null для обычного клиента
     * @param request     Запрос
     */
    @Nonnull
    public RegisterClientResult registerNewClient(
            Long operatorUid, @Nullable UidAndClientId agency, RegisterClientRequest request) {
        return registerNewClient(operatorUid, agency, request, false);
    }

    /**
     * Зарегистрировать нового клиента в балансе.
     * <p>
     * См. Clients.pm::CreateNewSubclient
     *
     * @param operatorUid           UID оператора
     * @param agency                Агентство для субклиента. Null для обычного клиента
     * @param request               Запрос
     * @param useExistingIfRequired использовать ClientId существующего клиента, если таковой есть
     */
    @Nonnull
    public RegisterClientResult registerNewClient(
            Long operatorUid, @Nullable UidAndClientId agency, RegisterClientRequest request,
            boolean useExistingIfRequired) {
        Objects.requireNonNull(operatorUid, "operatorUid");
        Objects.requireNonNull(request, "request");

        if (!request.isValid()) {
            return RegisterClientResult.error(RegisterClientStatus.INVALID_PARAMS);
        }

        try {
            boolean doCreateClientUidAssociation = true;
            if (useExistingIfRequired) {
                FindClientResult findClientResult =
                        findExistingClient(request.getClientUid(), request.getCountryRegionId(), request.getCurrency());
                if (findClientResult.isValidClientFound()) {
                    doCreateClientUidAssociation = false;
                    request.mergeWithFindClientResponseItem(findClientResult.getItem());
                } else if (findClientResult.isClientFoundButInvalid()) {
                    return RegisterClientResult.error(RegisterClientStatus.INVALID_CURRENCY);
                }
            }
            ClientId clientId = createClient(operatorUid, agency, request, false);
            if (clientId == null) {
                return RegisterClientResult.error(RegisterClientStatus.CANT_CREATE_CLIENT);
            }

            if (doCreateClientUidAssociation && !createClientAssociation(
                    operatorUid, UidAndClientId.of(request.getClientUid(), clientId))) {
                return RegisterClientResult.error(RegisterClientStatus.CANT_CREATE_CLIENT_ASSOCIATION);
            }

            return RegisterClientResult.success(clientId);
        } catch (RuntimeException e) {
            logger.error("Can't register agency client", e);
            return RegisterClientResult.error(RegisterClientStatus.UNKNOWN_ERROR);
        }
    }

    /**
     * Получить информацию по клиенту в балансе, если он есть
     *
     * @return Существующий идентификатор клиента в балансе, если он есть, null иначе
     */
    private FindClientResult findExistingClient(Long uid, @Nullable Long regionId,
                                                @Nullable CurrencyCode currency) {
        FindClientResponseItem item = findExistingClientByUid(uid);
        return item == null ? FindClientResult.clientNotFoundResult() :
                FindClientResult.clientFoundResult(item,
                        validateExistingClientCountryCurrency(item, regionId, currency));
    }

    @Nullable
    private FindClientResponseItem findExistingClientByUid(Long uid) {
        FindClientRequest request = new FindClientRequest()
                .withUid(uid)
                .withAgencySelectPolicy(AgencySelectPolicy.ASP_ALL)
                .withPrimaryClients(true);
        List<FindClientResponseItem> findClientResponse = balanceClient.findClient(request);
        return findClientResponse.stream().findFirst().orElse(null);
    }

    /**
     * Ищет clientId по UID'у.
     * Под капотом используется ручка Balance2.FindClient
     * Не должна падать, если клиент не найден.
     */
    public Optional<ClientId> findClientIdByUid(Long uid) {
        FindClientResponseItem item = findExistingClientByUid(uid);
        return Optional.ofNullable(item).map(FindClientResponseItem::getClientId).map(ClientId::fromLong);
    }

    /**
     * Ищет клиента по UID'у.
     * Под капотом используется ручка Balance2.FindClient
     * Не должна падать, если клиент не найден.
     */
    public Optional<FindClientResponseItem> findClientIdByUidFull(Long uid) {
        FindClientResponseItem item = findExistingClientByUid(uid);
        return Optional.ofNullable(item);
    }

    /**
     * Возвращает clientId по UID'у.
     * Под капотом используется ручка Balance2.GetPassportByUid
     * Падает с исключением, если клиент не найден.
     */
    public ClientId getClientIdByUid(Long uid) {
        ClientPassportInfo passportByUid = balanceClient.getPassportByUid(uid, uid);
        return ClientId.fromNullableLong(passportByUid.getClientId());
    }

    /**
     * Получить информацию о картах клиента
     *
     * @param uid            - uid пользователя
     * @param currencyFilter - фильтр по валюте
     * @return - список карт
     */
    public List<CardInfo> getClientPaymentCards(Long uid, String currencyFilter) {
        String userIp = getRemoteAddress()
                .map(InetAddress::getHostAddress)
                .orElse(LOCALHOST);

        return getUserCards(uid, userIp)
                .stream()
                .filter(x -> currencyFilter.equalsIgnoreCase(x.getCurrency()))
                .collect(toList());
    }

    public Set<PaymentMethodType> getUserPaymentMethodTypes(Long uid) {
        return getUserPaymentMethods(uid, false)
                .map(map -> map.get(PAYMENT_METHODS_TYPE_KEY))
                .filter(Objects::nonNull)
                .map(Object::toString)
                .map(PaymentMethodType::fromString)
                .toSet();
    }

    /**
     * Возвращает идентификаторы привязанных карт пользователя
     */
    public List<String> getUserCardIds(Long uid) {
        return getUserCards(uid).stream()
                .map(CardInfo::getCardId)
                .collect(toList());
    }

    /**
     * Возвращает информацию о привязанных картах пользователя
     */
    public List<CardInfo> getUserCards(Long uid) {
        return getUserCards(uid, LOCALHOST);
    }

    /**
     * Возвращает информацию о привязанных картах пользователя
     */
    public List<CardInfo> getUserCards(Long uid, String userIp) {
        return getUserPaymentMethods(uid, true, userIp)
                .filter(map -> map.get(PAYMENT_METHODS_TYPE_KEY).equals(PaymentMethodType.CARD.getBalanceName()))
                .map(m -> new CardInfo()
                        .withCardId((String) m.get("card_id"))
                        .withSystem((String) m.get("system"))
                        .withCurrency((String) m.get("currency"))
                        .withMaskedNumber((String) m.get("number"))
                )
                .toList();
    }

    private StreamEx<Map> getUserPaymentMethods(Long uid, boolean useBalanceSimple) {
        return getUserPaymentMethods(uid, useBalanceSimple, LOCALHOST);
    }

    private StreamEx<Map> getUserPaymentMethods(Long uid, boolean useBalanceSimple, String userIp) {
        ListPaymentMethodsSimpleRequest request = new ListPaymentMethodsSimpleRequest()
                .withUid(uid)
                .withUserIp(userIp)
                .withServiceToken(directServiceToken);
        ListPaymentMethodsSimpleResponseItem result = useBalanceSimple ?
                balanceClient.listPaymentMethods(request) : balanceClient.listPaymentMethodsSimple(request);
        if (result.getPaymentMethods() == null) {
            return StreamEx.empty();
        }

        return StreamEx.of(result.getPaymentMethods().values());
    }

    public boolean checkManagerExist(Long managerUid) {
        return balanceClient.massCheckManagersExist(singleton(managerUid))
                .contains(managerUid);
    }

    /**
     * Проверяем найденного клиента на соответствие валюте и региону
     *
     * @param item     найденный клиент
     * @param regionId id региона, с которым сверять клиента
     * @param currency код валюты, с которой сверять клиента
     * @return true, если валидация успешна, false иначе
     */
    public boolean validateExistingClientCountryCurrency(@Nonnull FindClientResponseItem item, @Nullable Long regionId,
                                                         @Nullable CurrencyCode currency) {
        if (regionId == null) {
            return true;
        }
        Collection<CurrencyCode> availableCurrencies = getClientCountryCurrencies(
                ClientId.fromNullableLong(item.getClientId()),
                ClientId.fromNullableLong(item.getAgencyId()))
                .get(regionId);
        availableCurrencies = nvl(availableCurrencies, emptyList());
        if (currency == null || availableCurrencies.stream().anyMatch(currency::equals)) {
            return true;
        }
        return false;
    }

    /**
     * Получить данные по странам и валютам для агенства (См. Common.pm::get_country_and_currency_from_balance)
     * <p>
     * Пока получение данных не кешируется
     *
     * @return Возвращает отображение идентификатора региона в список допустимых валют в данном регионе
     */
    @Nonnull
    public Multimap<Long, CurrencyCode> getClientCountryCurrencies(
            @Nullable ClientId clientId, @Nullable ClientId agencyId) {
        GetFirmCountryCurrencyRequest request = new GetFirmCountryCurrencyRequest().withCurrencyFilter(true);
        if (clientId != null) {
            request.withClientId(clientId.asLong());
        }
        if (agencyId != null) {
            request.withAgencyId(agencyId.asLong());
        }
        return balanceClient.getFirmCountryCurrency(request)
                .stream()
                .filter(r -> Currencies.currencyExists(r.getCurrency()))
                .collect(
                        toMultimap(
                                r -> r.getRegionId().longValue(),
                                r -> Currencies.getCurrency(r.getCurrency())
                                        .getCode()));
    }

    /**
     * Создает счет в Балансе и инициирует его оплату через форму траста. Возвращает сслыку на форму оплаты
     */
    public CreateAndPayRequestResult createAndPayRequest(Long operatorUid, ClientId clientId, Long campaignId,
                                                         CurrencyCode currencyCode, BigDecimal sum, Long personId,
                                                         @Nullable String redirectUrl, boolean forceUnmoderated,
                                                         String promocode, boolean denyPromocode) {
        CreateRequest2Item requestItem = new CreateRequest2Item()
                .withQty(sum)
                .withServiceId(directServiceId)
                .withServiceOrderId(campaignId);

        CreateRequest2Response response;
        try {
            response = balanceClient.createRequest(
                    operatorUid, clientId.asLong(), requestItem, forceUnmoderated, promocode, denyPromocode);
        } catch (BalanceClientException e) {
            return new CreateAndPayRequestResult(null, null, handleAndConvertBalanceException(promocode, e));
        }

        Long requestId = response.getRequestID();

        PayRequestRequest payRequestRequest = new PayRequestRequest()
                .withOperatorUid(operatorUid)
                .withRequestId(requestId)
                .withPaymentMethodId("trust_web_page")
                .withCurrency(currencyCode.name())
                .withPersonId(personId)
                .withRedirectUrl(redirectUrl);

        PayRequestResponse payRequestResponse;
        try {
            payRequestResponse = balanceClient.payRequest(payRequestRequest);
        } catch (BalanceClientException e) {
            return new CreateAndPayRequestResult(null, null, handleAndConvertBalanceException(promocode, e));
        }
        String paymentUrl = payRequestResponse.getPaymentUrl();
        BigDecimal amount = payRequestResponse.getAmount();

        return new CreateAndPayRequestResult(paymentUrl, amount, ValidationResult.success(new Object()));
    }

    public CreateInvoiceRequestResult createInvoice(Long operatorUid, ClientId clientId, Long campaignId,
                                                    Long paysysId, BigDecimal sum, Long personId,
                                                    boolean forceUnmoderated, String promocode, boolean denyPromocode) {
        CreateRequest2Item requestItem = new CreateRequest2Item()
                .withQty(sum)
                .withServiceId(directServiceId)
                .withServiceOrderId(campaignId);

        CreateRequest2Response createRequest2Response;
        try {
            createRequest2Response = balanceClient.createRequest(
                    operatorUid, clientId.asLong(), requestItem, forceUnmoderated, promocode, denyPromocode);
        } catch (BalanceClientException e) {
            return new CreateInvoiceRequestResult(null, handleAndConvertBalanceException(promocode, e));
        }

        Long requestId = createRequest2Response.getRequestID();

        CreateInvoiceRequest createInvoiceRequest = new CreateInvoiceRequest()
                .withOperatorUid(operatorUid)
                .withRequestId(requestId)
                .withPaysysId(paysysId)
                .withPersonId(personId)
                .withCredit(false);

        Long invoiceId;
        try {
            invoiceId = balanceClient.createInvoice(createInvoiceRequest);
        } catch (BalanceClientException e) {
            return new CreateInvoiceRequestResult(null, handleAndConvertBalanceException(promocode, e));
        }

        return new CreateInvoiceRequestResult(invoiceId, ValidationResult.success(new Object()));
    }

    private static ValidationResult<?, Defect> handleAndConvertBalanceException(
            String promocode, BalanceClientException e) throws BalanceClientException {
        if (nvl(e.getMessage(), "").contains("PROMOCODE_WRONG_CLIENT")) {
            ItemValidationBuilder<?, Defect> ib = ItemValidationBuilder.of(new Object());
            ItemValidationBuilder<String, Defect> promocodeVb = ib.item(promocode, "promocode");
            promocodeVb.check(Constraint.fromPredicateOfNullable(t -> false,
                    new Defect<>(BalanceDefectIds.PROMOCODE_WRONG_CLIENT)));
            return ib.getResult();
        } else if ((nvl(e.getMessage(), "").contains("ID_PC_NOT_UNIQUE_URLS"))) {
            ItemValidationBuilder<?, Defect> ib = ItemValidationBuilder.of(new Object());
            ItemValidationBuilder<String, Defect> promocodeVb = ib.item(promocode, "promocode");
            promocodeVb.check(Constraint.fromPredicateOfNullable(t -> false,
                    new Defect<>(BalanceDefectIds.UNIQUE_URLS_PROMOCODE)));
            return ib.getResult();
        } else if (nvl(e.getMessage(), "").contains("INVALID_PROMO_CODE")) { //спарсить XmlRpcException выше уровнем,
            // там внутри errorMessage-xml вида https://paste.yandex-team.ru/1016111
            ItemValidationBuilder<?, Defect> ib = ItemValidationBuilder.of(new Object());
            ItemValidationBuilder<String, Defect> promocodeVb = ib.item(promocode, "promocode");
            if (nvl(e.getMessage(), "").contains("INVALID_MINIMAL_QTY")) {
                BigDecimal minQuantity = null;
                Matcher matcher = PROMOCODE_MIN_VALUE_PATTERN.matcher(nvl(e.getMessage(), ""));
                if (matcher.find()) {
                    minQuantity = new BigDecimal(matcher.group(1));
                } else {
                    logger.error("cant detect min_qty in response " + nvl(e.getMessage(), ""));
                }
                promocodeVb.check(Constraint.fromPredicateOfNullable(t -> false,
                        new Defect<>(BalanceNumericDefectIds.INVALID_MINIMAL_QTY, minQuantity)));
            } else {
                promocodeVb.check(Constraint.fromPredicateOfNullable(t -> false,
                        new Defect<>(BalanceDefectIds.INVALID_PROMO_CODE)));
            }
            return ib.getResult();
        } else {
            throw e;
        }
    }

    @Nullable
    public GetClientPersonsResponseItem getPerson(ClientId clientId, Long personId) {
        var persons = balanceClient.getClientPersons(clientId.asLong());
        var person = persons.stream().filter(p -> p.getId().equals(personId)).findFirst();
        return person.orElse(null);
    }

    /**
     * Получаем плательщика из Баланса.
     * Если хотели физика, то создаем если их 0, берем любого, если их несколько
     * Если хотели юрика, то возвращаем null, если он не единственный (единственного иначе)
     */
    @Nullable
    public Long getOrCreatePerson(User user, Long operatorUid, boolean isLegalPerson) {
        List<GetClientPersonsResponseItem> persons = balanceClient.getClientPersons(user.getClientId().asLong());
        Map<String, List<GetClientPersonsResponseItem>> notHiddenPersons = StreamEx.of(persons)
                .filter(person -> nvl(person.getHidden(), 0).equals(0))
                .mapToEntry(GetClientPersonsResponseItem::getType, Function.identity())
                .grouping();

        if (isLegalPerson) {
            List<GetClientPersonsResponseItem> urPersons = notHiddenPersons.get(LEGAL_PERSON_TYPE);
            if (urPersons == null || urPersons.size() != 1) {
                return null;
            } else {
                return urPersons.get(0).getId();
            }
        } else {
            List<GetClientPersonsResponseItem> phPersons = notHiddenPersons.get(NATURAL_PERSON_TYPE);
            if (phPersons != null && !phPersons.isEmpty()) {
                return phPersons.get(0).getId();
            } else {
                return createNaturalPerson(user, operatorUid);
            }
        }
    }

    Long createNaturalPerson(User user, Long operatorUid) {

        CreatePersonRequest createPersonRequest = new CreatePersonRequest()
                .withOperatorUid(operatorUid)
                .withClientId(user.getClientId().toString())
                .withType(NATURAL_PERSON_TYPE);

        String fio = user.getFio().trim();
        String[] fioParts = fio.split("\\s+", 2);

        if (fio.isBlank()) {
            createPersonRequest.withFirstName(DEFAULT_FIRST_NAME);
            createPersonRequest.withLastName(DEFAULT_LAST_NAME);
        } else if (fioParts.length == 1) {
            createPersonRequest.withFirstName(DEFAULT_FIRST_NAME);
            createPersonRequest.withLastName(fio);
        } else {
            createPersonRequest.withFirstName(fioParts[0]);
            createPersonRequest.withLastName(fioParts[1]);
        }
        createPersonRequest.withMiddlename(DEFAULT_FIRST_NAME);

        createPersonRequest.withEmail(user.getEmail());
        createPersonRequest.withPhone(nvl(user.getPhone(), DEFAULT_PHONE));

        return balanceClient.createPerson(createPersonRequest);
    }

    public GetCardBindingURLResponse getCardBinding(Long operatorUid, CurrencyCode currency,
                                                    @Nullable String returnPath, boolean isMobileForm) {
        return balanceClient.getCardBindingURL(
                operatorUid, currency.name(), returnPath, directServiceId, isMobileForm);
    }

    /**
     * Пакетное создание/обновление заказов в Балансе, смотри yandex-lib/balance/lib/Yandex/Balance.pm:387
     *
     * @return возвращает true при успешном выполнении запрос и false если есть ошибки
     */
    public boolean createOrUpdateOrders(CreateOrUpdateOrdersBatchRequest request) {
        request.setServiceToken(directServiceToken);
        try {
            List<List<String>> response = balanceClient.createOrUpdateOrdersBatch(request);
            return response.stream()
                    .map(responseItem -> responseItem.get(0))
                    .allMatch(responseItemFirstPart -> Objects.equals("0", responseItemFirstPart));
        } catch (BalanceClientException e) {
            return false;
        }
    }

    public CheckBindingResponse checkBinding(Long operatorUid, String purchaseToken) {
        CheckBindingRequest request = new CheckBindingRequest()
                .withOperatorUid(operatorUid)
                .withServiceId(directServiceId)
                .withPurchaseToken(purchaseToken);

        return balanceClient.checkBinding(request);
    }

    public void createTransferMultiple(CreateTransferMultipleRequest request) {
        balanceClient.createTransferMultiple(request);
    }
}
