package ru.yandex.direct.core.service.integration.passport;

import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Lazy;
import org.springframework.retry.RetryOperations;
import org.springframework.retry.backoff.UniformRandomBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.solomon.SolomonExternalSystemMonitorService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.inside.passport.internal.api.PassportClient;
import ru.yandex.inside.passport.internal.api.YaHeaders;
import ru.yandex.inside.passport.internal.api.exceptions.RecoverablePassportClientException;
import ru.yandex.inside.passport.internal.api.models.Language;
import ru.yandex.inside.passport.internal.api.models.phone.PhoneCommitRequest;
import ru.yandex.inside.passport.internal.api.models.phone.PhoneSubmitRequest;
import ru.yandex.inside.passport.internal.api.models.phone.PhoneSubmitResponse;
import ru.yandex.inside.passport.internal.api.models.registration.RegistrationRequest;
import ru.yandex.inside.passport.internal.api.models.registration.RegistrationResponse;
import ru.yandex.inside.passport.internal.api.models.supportcode.CreateSupportCodeRequest;
import ru.yandex.inside.passport.internal.api.models.supportcode.CreateSupportCodeResponse;
import ru.yandex.inside.passport.internal.api.models.validation.ValidatePhoneNumberResponse;

import static java.util.Collections.singletonMap;
import static java.util.Objects.nonNull;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_2XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_4XX;
import static ru.yandex.direct.solomon.SolomonResponseMonitorStatus.STATUS_5XX;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;

/**
 * Интеграция с паспортом
 */
@ParametersAreNonnullByDefault
@Lazy
public class PassportService {
    private static final int DEFAULT_PASSWORD_SIZE = 20;
    private static final Language DEFAULT_LANGUAGE = Language.RU;
    private static final String DEFAULT_COUNTRY = "RU";
    private static final String FAKE_USERAGENT = "Direct fake user agent for server-side calls";

    // описание статуса есть в https://wiki.yandex-team
    // .ru/passport/api/bundle/registration/?from=%252Fpassport%252Fpython%252Fapi%252Fbundle%252Fregistration%252F
    // #registracijalajjtasalternativnojjvalidaciejjkapchailitelefon
    static final String ACCOUNT_ALREADY_REGISTERED_ERROR = "login.notavailable";

    static final String DEFAULT_EMAIL_DOMAIN = "yandex.ru";

    private static final Logger logger = LoggerFactory.getLogger(PassportService.class);

    private static final String EXTERNAL_SYSTEM = "passport";
    private static final String METHOD_SUBMIT = "confirmAndBindSubmit";
    private static final String METHOD_COMMIT = "confirmAndBindCommit";
    private static final String METHOD_REGISTER_USER = "registerUser";
    private static final String METHOD_SUPPORT_CODES = "supportCodes";

    private static final SolomonExternalSystemMonitorService monitorService = new SolomonExternalSystemMonitorService(
            EXTERNAL_SYSTEM,
            Set.of(METHOD_SUBMIT, METHOD_COMMIT, METHOD_REGISTER_USER, METHOD_SUPPORT_CODES)
    );

    private final PassportClient passportClient;
    private final String consumer;
    private final RetryOperations retryOperations;

    public PassportService(
            PassportClient passportClient,
            String consumer,
            int retryMaxAttempts,
            int retryBackoffDelayInMs) {
        Objects.requireNonNull(passportClient, "passportClient");
        Objects.requireNonNull(consumer, "consumer");
        if (retryMaxAttempts <= 0) {
            throw new IllegalArgumentException("retryMaxAttempts: " + retryMaxAttempts);
        }
        if (retryBackoffDelayInMs < 0) {
            throw new IllegalArgumentException("retryBackoffDelayInMs: " + retryBackoffDelayInMs);
        }

        this.passportClient = passportClient;
        this.consumer = consumer;

        RetryTemplate retryTemplate = new RetryTemplate();
        retryTemplate.setRetryPolicy(
                new SimpleRetryPolicy(
                        retryMaxAttempts,
                        singletonMap(RecoverablePassportClientException.class, true)));

        if (retryBackoffDelayInMs > 0) {
            UniformRandomBackOffPolicy backOffPolicy = new UniformRandomBackOffPolicy();
            backOffPolicy.setMinBackOffPeriod(0);
            backOffPolicy.setMaxBackOffPeriod(retryBackoffDelayInMs);
            retryTemplate.setBackOffPolicy(backOffPolicy);
        }
        this.retryOperations = retryTemplate;
    }

    /**
     * Зарегистрировать пользователя в паспорте
     */
    public RegisterUserResult registerUser(
            String transactionId,
            RegisterUserRequest request,
            String remoteClientIp
    ) {
        String password = PassportUtils.generateRandomPassword(DEFAULT_PASSWORD_SIZE);
        var response = doRegisterUser(transactionId, request, remoteClientIp, password);

        writeMetrics(response, r -> !r.isSuccess() || !isEmpty(r.getErrors()), METHOD_REGISTER_USER);

        if (response == null) {
            return RegisterUserResult.error(RegisterUserStatus.INTERNAL_ERROR);
        }
        if (!response.isSuccess()) {
            logger.info("Got passport errors during registerUser: {}", response.getErrors());

            if (response.getErrors().contains(ACCOUNT_ALREADY_REGISTERED_ERROR)) {
                return RegisterUserResult.error(RegisterUserStatus.LOGIN_OCCUPIED);
            } else {
                return RegisterUserResult.error(RegisterUserStatus.REQUEST_FAILED);
            }
        }
        return RegisterUserResult.success(
                Long.parseLong(response.getUid()),
                request.getLogin() + "@" + DEFAULT_EMAIL_DOMAIN,
                password);
    }

    /**
     * Зарегистрировать пользователя в паспорте
     */
    @Nullable
    private RegistrationResponse doRegisterUser(
            String transactionId,
            RegisterUserRequest request,
            String remoteClientIp,
            String password
    ) {
        try {
            return retryOperations.execute(context -> {
                try (TraceProfile ignore = Trace.current().profile("passport", "registerByMiddleman")) {
                    return passportClient.accounts()
                            .registerByMiddleman(
                                    transactionId,
                                    consumer,
                                    new RegistrationRequest()
                                            .withLogin(request.getLogin())
                                            .withPassword(password)
                                            .withFirstName(request.getFirstName())
                                            .withLastName(request.getLastName())
                                            .withLanguage(DEFAULT_LANGUAGE)
                                            .withCountry(DEFAULT_COUNTRY),
                                    new YaHeaders().withYaClientUserAgent(FAKE_USERAGENT)
                                            .withYaConsumerClientIp(remoteClientIp));
                }
            });
        } catch (RuntimeException e) {
            logger.error("Error while registerUser", e);
            return null;
        }
    }

    @Nullable
    public PhoneSubmitResponse confirmAndBindSubmit(String requestId, String phoneNumber) {
        PhoneSubmitResponse submitResponse = doConfirmAndBindSubmit(requestId, phoneNumber);
        writeMetrics(submitResponse, r -> !isEmpty(r.getErrors()), METHOD_SUBMIT);
        return submitResponse;
    }

    private PhoneSubmitResponse doConfirmAndBindSubmit(String requestId, String phoneNumber) {
        Locale locale = HttpUtil.getCurrentLocale().orElse(Locale.ENGLISH);
        try {
            return passportClient.phones().confirmAndBindSubmit(requestId, consumer,
                    new PhoneSubmitRequest()
                            .withDisplayLanguage(locale.getLanguage())
                            .withNumber(phoneNumber),
                    buildDefaultHeaders());
        } catch (Exception e) {
            logger.error("Got passport internal error during submit", e);
            return null;
        }
    }

    @Nullable
    public ValidatePhoneNumberResponse confirmAndBindCommit(String requestId, String trackId, String code) {
        ValidatePhoneNumberResponse commitResponse = doConfirmAndBindCommit(requestId, trackId, code);
        writeMetrics(commitResponse, r -> !isEmpty(r.getErrors()), METHOD_COMMIT);
        return commitResponse;
    }

    private ValidatePhoneNumberResponse doConfirmAndBindCommit(String requestId, String trackId, String code) {
        try {
            return passportClient.phones().confirmAndBindCommit(requestId, consumer,
                    new PhoneCommitRequest()
                            .withTrackId(trackId)
                            .withCode(code),
                    buildDefaultHeaders());
        } catch (Exception e) {
            logger.error("Got passport internal error during commit", e);
            return null;
        }
    }

    @Nullable
    public CreateSupportCodeResponse createSupportCode(String requestId, Long uid, String oAuthToken) {
        CreateSupportCodeResponse codeResponse = doCreateSupportCode(requestId, uid, oAuthToken);
        writeMetrics(codeResponse, r -> !isEmpty(r.getErrors()), METHOD_SUPPORT_CODES);
        return codeResponse;
    }

    @Nullable
    public CreateSupportCodeResponse doCreateSupportCode(String requestId, Long uid, String oAuthToken) {
        try {
            return passportClient.supportCodes().create(requestId, consumer,
                    new CreateSupportCodeRequest().withUid(uid),
                    buildDefaultHeaders().withYaConsumerAuthorization(oAuthToken));
        } catch (RuntimeException e) {
            logger.error("Got passport internal error during support code create", e);
            return null;
        }
    }

    private YaHeaders buildDefaultHeaders() {
        String remoteIp = HttpUtil.getRemoteAddress().orElseThrow(IllegalStateException::new).getHostAddress();
        String cookies = HttpUtil.getRequest().getHeader("Cookie");
        String host = HttpUtil.getRequest().getHeader("Host");

        var headers = new YaHeaders()
                .withYaConsumerClientScheme("https")
                .withYaConsumerClientIp(remoteIp)
                .withYaClientUserAgent(FAKE_USERAGENT)
                .withYaClientHost(host);

        if (nonNull(cookies)) {
            headers.withYaClientCookie(cookies);
        }

        return headers;
    }

    private <T> void writeMetrics(@Nullable T response, Predicate<T> hasErrors, String methodName) {
        if (response == null) {
            monitorService.write(methodName, STATUS_5XX);
        } else if (hasErrors.test(response)) {
            monitorService.write(methodName, STATUS_4XX);
        } else {
            monitorService.write(methodName, STATUS_2XX);
        }
    }
}
