package ru.yandex.direct.core.service.integration.passport;

import java.io.IOException;
import java.net.URI;
import java.util.Objects;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.protocol.HttpContext;

import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.inside.passport.internal.api.PassportClient;
import ru.yandex.inside.passport.tvm2.TvmHeaders;

public class TvmPassportClientBuilder {

    private URI serviceUri;
    private Integer maxConnectionCount;
    private Integer connectionRequestTimeoutInMs;
    private Integer connectTimeoutInMs;
    private Integer socketTimeoutInMs;
    private HttpRequestRetryHandler retryHandler;
    private TvmIntegration tvmIntegration;
    private TvmService tvmService;

    @Override
    public String toString() {
        return "PassportClientBuilder{" +
                "serviceUri=" + serviceUri +
                ", maxConnectionCount=" + maxConnectionCount +
                ", connectionRequestTimeoutInMs=" + connectionRequestTimeoutInMs +
                ", connectTimeoutInMs=" + connectTimeoutInMs +
                ", socketTimeoutInMs=" + socketTimeoutInMs +
                '}';
    }

    public URI getServiceUri() {
        return serviceUri;
    }

    public TvmPassportClientBuilder withServiceUri(URI serviceUri) {
        this.serviceUri = Objects.requireNonNull(serviceUri, "serviceUri");
        return this;
    }

    public Integer getMaxConnectionCount() {
        return maxConnectionCount;
    }

    public TvmPassportClientBuilder withMaxConnectionCount(int maxConnectionCount) {
        if (maxConnectionCount <= 0) {
            throw new IllegalArgumentException("maxConnPerRoute: " + maxConnectionCount);
        }
        this.maxConnectionCount = maxConnectionCount;
        return this;
    }

    public int getConnectionRequestTimeoutInMs() {
        return connectionRequestTimeoutInMs;
    }

    public TvmPassportClientBuilder withConnectionRequestTimeoutInMs(int connectionRequestTimeoutInMs) {
        if (connectionRequestTimeoutInMs <= 0) {
            throw new IllegalArgumentException("connectionRequestTimeoutInMs: " + connectionRequestTimeoutInMs);
        }
        this.connectionRequestTimeoutInMs = connectionRequestTimeoutInMs;
        return this;
    }

    public Integer getConnectTimeoutInMs() {
        return connectTimeoutInMs;
    }

    public TvmPassportClientBuilder withConnectTimeoutInMs(int connectTimeoutInMs) {
        if (connectTimeoutInMs <= 0) {
            throw new IllegalArgumentException("connectTimeoutInMs: " + connectTimeoutInMs);
        }
        this.connectTimeoutInMs = connectTimeoutInMs;
        return this;
    }

    public Integer getSocketTimeoutInMs() {
        return socketTimeoutInMs;
    }

    public TvmPassportClientBuilder withSocketTimeoutInMs(int socketTimeoutInMs) {
        if (socketTimeoutInMs <= 0) {
            throw new IllegalArgumentException("socketTimeoutInMs: " + socketTimeoutInMs);
        }
        this.socketTimeoutInMs = socketTimeoutInMs;
        return this;
    }

    public HttpRequestRetryHandler getRetryHandler() {
        return retryHandler;
    }

    public TvmPassportClientBuilder withRetryHandler(HttpRequestRetryHandler retryHandler) {
        this.retryHandler = retryHandler;
        return this;
    }

    public TvmIntegration getTvmIntegration() {
        return tvmIntegration;
    }

    public TvmPassportClientBuilder withTvmIntegration(TvmIntegration tvmIntegration) {
        this.tvmIntegration = tvmIntegration;
        return this;
    }

    public TvmService getTvmServiceId() {
        return tvmService;
    }

    public TvmPassportClientBuilder withTvmServiceId(TvmService tvmService) {
        this.tvmService = tvmService;
        return this;
    }

    public PassportClient build() {
        if (connectionRequestTimeoutInMs <= 0) {
            throw new IllegalArgumentException("connectionRequestTimeoutInMs: " + connectionRequestTimeoutInMs);
        }
        if (connectTimeoutInMs <= 0) {
            throw new IllegalArgumentException("connectTimeoutInMs: " + connectTimeoutInMs);
        }
        if (socketTimeoutInMs <= 0) {
            throw new IllegalArgumentException("socketTimeoutInMs: " + socketTimeoutInMs);
        }
        if (maxConnectionCount <= 0) {
            throw new IllegalArgumentException("maxConnectionCount: " + maxConnectionCount);
        }

        RequestConfig.Builder requestConfigBuilder = RequestConfig.copy(RequestConfig.DEFAULT);

        if (connectionRequestTimeoutInMs != null) {
            requestConfigBuilder.setConnectionRequestTimeout(connectionRequestTimeoutInMs);
        }

        if (connectTimeoutInMs != null) {
            requestConfigBuilder.setConnectTimeout(connectTimeoutInMs);
        }

        if (socketTimeoutInMs != null) {
            requestConfigBuilder.setSocketTimeout(socketTimeoutInMs);
        }

        HttpClientBuilder httpClientBuilder = HttpClients.custom();

        if (maxConnectionCount != null) {
            httpClientBuilder.setMaxConnPerRoute(maxConnectionCount)
                    .setMaxConnTotal(maxConnectionCount);
        }

        if (retryHandler != null) {
            httpClientBuilder.setRetryHandler(retryHandler);
        }

        httpClientBuilder.setDefaultRequestConfig(requestConfigBuilder.build());

        if (tvmIntegration != null) {
            httpClientBuilder.addInterceptorFirst(new TvmInterceptor(tvmIntegration, tvmService));
        }

        return new PassportClient(httpClientBuilder.build(), true, serviceUri);
    }

    private static class TvmInterceptor implements HttpRequestInterceptor {

        private final TvmIntegration tvmIntegration;
        private final TvmService passportService;

        private TvmInterceptor(TvmIntegration tvm2, TvmService passportService) {
            this.tvmIntegration = tvm2;
            this.passportService = passportService;
        }

        @Override
        public void process(HttpRequest request, HttpContext context) throws HttpException, IOException {
            String serviceTicket = tvmIntegration.getTicket(passportService);
            if (serviceTicket != null) {
                request.addHeader(TvmHeaders.SERVICE_TICKET, serviceTicket);
            }
        }
    }

}
