package ru.yandex.direct.core.service.storeurlchecker;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.core.entity.zora.ZoraService;
import ru.yandex.direct.zorafetcher.ZoraResponse;

/**
 * Компонент, умеющий сходить и проверить всё ли ОК со страницей по указанному урлу.
 * Реально. По-настоящему. Через интернет (посредством Zora).
 */
@Component
@ParametersAreNonnullByDefault
public class StoreUrlInstantChecker {

    private static final Logger logger = LoggerFactory.getLogger(StoreUrlInstantChecker.class);

    private final ZoraService zoraService;

    @Autowired
    public StoreUrlInstantChecker(ZoraService zoraService) {
        this.zoraService = zoraService;
    }

    /**
     * Сходить по адресу {@code storeUrl} и проверить есть ли там что-то осмысленное.
     * <p>
     * Ожидается, что {@code storeUrl} провалидирован, и выглядит как адрес страницы приложения в сторе.
     * Следует редиректам (не более 5-ти).
     *
     * @param storeUrl адрес приложения в сторе который нужно проверить
     * @return true, если есть HTTP 200 OK ответ от стора
     */
    public boolean isStoreUrlReachable(String storeUrl) {
        return isStoreUrlReachable(Collections.singletonList(storeUrl)).get(storeUrl);
    }

    Map<String, Boolean> isStoreUrlReachable(List<String> storeUrls) {
        logger.debug("urls to check: {}", storeUrls);

        Map<String, Result<ZoraResponse>> responses = zoraService.fetchByUrl(storeUrls, true);
        return EntryStream.of(responses)
                .mapValues(v -> v.getSuccess().isOk())
                .toMap();
    }
}
