package ru.yandex.direct.core.service.urlchecker;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.httpclient.util.URIUtil;
import org.apache.commons.lang3.StringUtils;
import org.asynchttpclient.uri.Uri;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.core.entity.banner.type.href.BannersUrlHelper;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.core.entity.zora.ZoraService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.zorafetcher.ZoraRequest;
import ru.yandex.direct.zorafetcher.ZoraResponse;

import static io.netty.handler.codec.http.HttpHeaderNames.LOCATION;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.constraint.StringConstraints.isValidHref;

/**
 * По урлу получить цепочку редиректов
 */
@Component
@ParametersAreNonnullByDefault
public class RedirectChecker {
    private static final long MAX_REQUEST_TIMEOUT = 10000;

    private static final Pattern META_REDIRECT_PATTERN = Pattern.compile(
            "\\s*0*(\\d+)\\s*;(?:\\s*url\\s*=)?\\s*[\"']?(\\w+://\\S+)[\"']?\\s*",
            Pattern.CASE_INSENSITIVE);

    private static final Logger logger = LoggerFactory.getLogger(RedirectChecker.class);

    private final TrustedRedirectsService trustedRedirectsService;
    private final BannersUrlHelper bannersUrlHelper;
    private final ZoraService zoraService;

    public RedirectChecker(TrustedRedirectsService trustedRedirectsService,
                           BannersUrlHelper bannersUrlHelper,
                           ZoraService zoraService) {
        this.trustedRedirectsService = trustedRedirectsService;
        this.bannersUrlHelper = bannersUrlHelper;
        this.zoraService = zoraService;
    }

    public RedirectCheckResult getRedirectUrl(UrlToCheck url) {
        GetRedirectChainResult result = getRedirectChain(url);
        if (!result.isSuccessful() || result.getRedirectChain().isEmpty()) {
            return RedirectCheckResult.createFailResult();
        }

        String lastUrl = Iterables.getLast(result.getRedirectChain());
        if (StringUtils.isBlank(lastUrl) || !isValidHref(lastUrl)) {
            return RedirectCheckResult.createFailResult();
        }

        String redirectUrl = bannersUrlHelper.toUnicodeUrl(lastUrl);
        String redirectDomain = bannersUrlHelper.extractHostFromHrefWithWwwOrNull(redirectUrl);

        return RedirectCheckResult.createSuccessResult(redirectUrl, redirectDomain);
    }

    public GetRedirectChainResult getRedirectChain(UrlToCheck url) {
        long remainedTimeout = url.getTimeout();
        long remainedRedirects = url.getRedirectsLimit();
        String nextUrl = url.getUrl();
        String userAgent = url.getUserAgent();
        boolean trustRedirectFromAnyDomain = nvl(url.getTrustRedirectFromAnyDomain(), false);
        List<String> redirectChain = new ArrayList<>();

        boolean useGoZora = zoraService.useGoZora();
        String tvmTicket = zoraService.getTvmTicket(useGoZora);

        try (
                TraceProfile traceProfile = Trace.current().profile("zora", "getRedirectChain")
        ) {
            long requestId = 0;
            while (remainedTimeout > 0 && remainedRedirects > 0) {
                if (!trustRedirectFromAnyDomain && !isKnownRedirect(nextUrl)) {
                    logger.info("Unknown redirect {}", nextUrl);
                    redirectChain.add(nextUrl);
                    return GetRedirectChainResult.createSuccessResult(redirectChain);
                }

                TraceChild traceChild = Trace.current().child("zora", "get");
                // TODO: использовать прокси из скриптов в режиме 'userproxy' нельзя
                ZoraRequest request = zoraService.createZoraRequest(++requestId,
                        zoraService.createGetRequest(nextUrl,
                                traceToHeader(traceChild),
                                Duration.ofMillis(Math.min(remainedTimeout, MAX_REQUEST_TIMEOUT)),
                                tvmTicket, false, useGoZora, userAgent), useGoZora);

                long started = System.currentTimeMillis();
                Result<ZoraResponse> result = zoraService.fetch(request, useGoZora);
                redirectChain.add(nextUrl);

                if (result.getErrors() != null) {
                    logger.warn("Received fetch result with errors: {}", result);
                    // в случае проблем с подключением к zora считаем, что url доступен
                    return GetRedirectChainResult.createSuccessResult(singletonList(url.getUrl()));
                }

                ZoraResponse res = result.getSuccess();

                // редирект
                if (res.isRedirect()) {
                    remainedTimeout -= System.currentTimeMillis() - started;
                    remainedRedirects--;
                    nextUrl = extractLocation(nextUrl, res);
                    continue;
                }

                // таймаут
                if (result.isTimeout()) {
                    logger.info("Zora return timeout for {}", nextUrl);
                    logZoraResponse(res);
                    return GetRedirectChainResult.createFailResult(redirectChain);
                }

                if (res.isOk()) {
                    // Сайт нашёлся и ответил ОК
                    Optional<String> metaRedirect = parseMetaRedirect(res.getResponse().getResponseBody());
                    if (metaRedirect.isPresent()) {
                        remainedTimeout -= System.currentTimeMillis() - started;
                        remainedRedirects--;
                        nextUrl = metaRedirect.get();
                        continue;
                    }
                    return GetRedirectChainResult.createSuccessResult(redirectChain);
                }

                logger.info("not OK for {}", nextUrl);
                logZoraResponse(res);
                return GetRedirectChainResult.createFailResult(redirectChain);
            }
        }

        if (remainedTimeout <= 0) {
            logger.info("timeout for {}", url.getUrl());
            return GetRedirectChainResult.createFailResult(redirectChain);
        } else {
            logger.info("redirect limit exceeded for {}", url.getUrl());
            return GetRedirectChainResult.createFailResult(redirectChain);
        }
    }

    /**
     * по HTTP::Response попробовать найти meta-редирект
     * (особый вид редиректа, располагается в заголовке html-документа)
     * <p>
     * <meta http-equiv="refresh" content="seconds;URL-to-redirect">
     * Первый параметр seconds - это количество секунд (после полной загрузки страницы), по истечении которых
     * произойдет редирект на второй параметр URL-to-redirect.
     */
    static Optional<String> parseMetaRedirect(@Nullable String responseBody) {
        try {
            return parseMetaRedirectUnsafe(responseBody);
        } catch (RuntimeException e) {
            logger.warn("can't get redirect from meta tags", e);
            return Optional.empty();
        }
    }

    private static Optional<String> parseMetaRedirectUnsafe(@Nullable String responseBody) {
        if (StringUtils.isEmpty(responseBody)) {
            return Optional.empty();
        }
        String htmlString = StringUtils.substring(responseBody, 0, 10_000);

        Document document = Jsoup.parse(htmlString);
        Elements elements = document.getElementsByTag("meta");

        for (Element element : elements) {
            if (StringUtils.containsIgnoreCase(element.attr("http-equiv"), "refresh")
                    || StringUtils.containsIgnoreCase(element.attr("name"), "refresh")) {

                String content = element.attr("content");
                Matcher matcher = META_REDIRECT_PATTERN.matcher(content);

                if (matcher.find()) {
                    String urlToRedirect = matcher.group(2);
                    long seconds = Long.parseLong(matcher.group(1));

                    if ((seconds <= 60) && StringUtils.isNotBlank(urlToRedirect) && isValidHref(urlToRedirect)) {
                        return Optional.of(urlToRedirect);
                    }
                }
            }
        }
        return Optional.empty();
    }

    /**
     * Возвращет полный URL из заголовка {@code Location} ответа {@code response}
     *
     * @param contextUrl базовый URL для относительных редиректов
     */
    public static String extractLocation(String contextUrl, ZoraResponse response) {
        String locationUrl = Uri.create(Uri.create(contextUrl), response.getResponse().getHeader(LOCATION)).toUrl();
        return fixRedirectLocationUrlIfNeeded(locationUrl);
    }

    private static String fixRedirectLocationUrlIfNeeded(String url) {
        if (isValidHref(url)) {
            return url;
        }

        String fixed = null;
        try {
            // Некоторые счётчики пытаются отдавать Location в кодировке ISO-8859-1: DIRECT-108816
            fixed = URIUtil.encodeQuery(url, StandardCharsets.ISO_8859_1.toString());
        } catch (URIException e) {
            // do nothing
        }
        if (fixed != null && isValidHref(fixed)) {
            return fixed;
        }

        if (url.startsWith("market://details")) {
            fixed = url.replace("market://details", "https://play.google.com/store/apps/details");
        } else if (url.startsWith("itms-apps://")) {
            fixed = url.replace("itms-apps://", "https://");
        } else if (url.startsWith("itms-appss://")) {
            fixed = url.replace("itms-appss://", "https://");
        }
        if (fixed != null && isValidHref(fixed)) {
            return fixed;
        }

        logger.warn("Can't fix redirect Location URL {}", url);
        return url;
    }

    private boolean isKnownRedirect(String url) {
        return trustedRedirectsService.isTrusted(url);
    }

    private void logZoraResponse(ZoraResponse res) {
        if (logger.isDebugEnabled()) {
            logger.debug(res.prettyZoraResponse());
        }
    }
}
