package ru.yandex.direct.core.service.urlchecker;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Chars;
import org.asynchttpclient.uri.Uri;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.core.entity.zora.ZoraService;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.UrlUtils;
import ru.yandex.direct.zorafetcher.ZoraRequest;
import ru.yandex.direct.zorafetcher.ZoraResponse;

import static io.netty.handler.codec.http.HttpHeaderNames.LOCATION;
import static java.util.Collections.unmodifiableSet;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants.TEMPLATE_LABEL_RE;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;

/**
 * Компонент, умеющий сходить и проверить всё ли ОК со страницей по указанному урлу (посредством Zora).
 * Учитывается допустимый timeout и максимальное количество редиректов.
 */
@Component
@ParametersAreNonnullByDefault
public class UrlChecker {
    private static final long MAX_REQUEST_TIMEOUT = 10000;
    private static final Pattern TEMPLATE_PATTERN = Pattern.compile(TEMPLATE_LABEL_RE);
    private static final Pattern USER_PARAMS_PATTERN = Pattern.compile(
            "\\{(?:CampaignID|BannerID|AdID|AdGroupID|param1|param2)}");

    private static final Logger logger = LoggerFactory.getLogger(UrlChecker.class);

    private static Set<String> brokenEncodingCases;
    private final ZoraService zoraService;

    public UrlChecker(ZoraService zoraService) {
        this.zoraService = zoraService;
    }

    public UrlCheckResult isUrlReachable(UrlToCheck url) {
        long remainedTimeout = url.getTimeout();
        long remainedRedirects = url.getRedirectsLimit();

        String nextUrl = url.getUrl();
        // replace template with its contents (#template# -> template)
        Matcher m = TEMPLATE_PATTERN.matcher(nextUrl);
        if (m.find()) {
            nextUrl = m.replaceAll("$1");
        }

        nextUrl = USER_PARAMS_PATTERN.matcher(nextUrl).replaceAll("");

        boolean useGoZora = zoraService.useGoZora();
        String tvmTicket = zoraService.getTvmTicket(useGoZora);

        try (
                TraceProfile traceProfile = Trace.current().profile("zora", "isUrlReachable")
        ) {
            long requestId = 0;
            while (remainedTimeout > 0 && remainedRedirects > 0) {
                TraceChild traceChild = Trace.current().child("zora", "get");
                ZoraRequest request = zoraService.createZoraRequest(++requestId,
                        zoraService.createGetRequest(nextUrl,
                                traceToHeader(traceChild),
                                Duration.ofMillis(Math.min(remainedTimeout, MAX_REQUEST_TIMEOUT)),
                                tvmTicket, false, useGoZora), useGoZora);

                long started = System.currentTimeMillis();
                Result<ZoraResponse> result = zoraService.fetch(request, useGoZora);
                if (result.getErrors() != null) {
                    logger.warn("Received fetch result with errors: {}", result);
                    // в случае проблем с подключением к zora считаем, что url доступен
                    return createSuccessResult();
                }
                ZoraResponse res = result.getSuccess();

                // редирект
                if (res.isRedirect()) {
                    remainedTimeout -= System.currentTimeMillis() - started;
                    remainedRedirects--;
                    nextUrl = Uri.create(Uri.create(nextUrl), res.getResponse().getHeader(LOCATION)).toUrl();
                    nextUrl = fixIfBrokenEncoding(nextUrl);
                    continue;
                }

                // таймаут
                if (res.isTimeout(result)) {
                    logger.info("Zora return timeout for {}", nextUrl);
                    logZoraResponse(res);
                    return createFailResult(UrlCheckResult.Error.TIMEOUT);
                }

                if (res.isOk()) {
                    // Сайт нашёлся и ответил ОК
                    return createSuccessResult();
                }

                logger.info("not OK for {}", nextUrl);
                logZoraResponse(res);
                return createFailResult(UrlCheckResult.Error.HTTP_ERROR);
            }
        }

        if (remainedTimeout <= 0) {
            logger.info("timeout for {}", url.getUrl());
            return createFailResult(UrlCheckResult.Error.TIMEOUT);
        } else {
            logger.info("redirect limit exceeded for {}", url.getUrl());
            return createFailResult(UrlCheckResult.Error.TOO_MANY_REDIRECTS);
        }
    }

    private void logZoraResponse(ZoraResponse res) {
        if (logger.isDebugEnabled()) {
            logger.debug(res.prettyZoraResponse());
        }
    }

    private UrlCheckResult createSuccessResult() {
        return new UrlCheckResult(true, null);
    }

    private UrlCheckResult createFailResult(UrlCheckResult.Error error) {
        return new UrlCheckResult(false, error);
    }

    /**
     * Обнаруживает и фиксит ситуацию, когда внешний сервер возвращает для редиректа ссылку
     * содержащую кирилицу просто в кодировке UTF-8, вместо положенного по стандарту енкодинга урла.
     */
    private static String fixIfBrokenEncoding(String srcUrl) {
        if (brokenEncodingCases == null) {
            brokenEncodingCases = getBrokenEncodingCases();
        }
        char[] chars = srcUrl.toCharArray();
        for (int i = 1; i < chars.length; i++) {
            String testStr = srcUrl.substring(i - 1, i + 1);
            if (brokenEncodingCases.contains(testStr)) {
                byte[] srcBytes = srcUrl.getBytes(StandardCharsets.ISO_8859_1);
                String decodedUrl = new String(srcBytes, StandardCharsets.UTF_8);
                return UrlUtils.encodeUrlIfCan(decodedUrl);
            }
        }
        return srcUrl;
    }

    /**
     * Генерирует символьные комбинации возникающие при разборе UTF-8 строки с помощью ISO-8859-1 кодировки.
     *
     * Если в будущем захочется поддержать что-то кроме UTF-8, то можно заменить сет на мапу с символьной комбинацией
     * в ключе и исходной кодировкой в значении. Но если комбинации станут не двухсимвольными, то алгоритм сравнения
     * тоже надо не забыть доработать.
     */
    private static Set<String> getBrokenEncodingCases() {
        char[] ca = "ЁАБВГДЕЖЗИЙКЛМНОПРСТУФХЦЧШЩЪЫЬЭЮЯабвгдежзийклмнопрстуфхцчшщъыьэюяё".toCharArray();
        Set<String> set = new HashSet<>();
        Chars.asList(ca).stream()
                .map(String::valueOf)
                .map(s -> new String(s.getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1))
                .forEach(set::add);
        return unmodifiableSet(set);
    }

}
