package ru.yandex.direct.core.smsauth.service;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.smsauth.model.SmsAuthKey;
import ru.yandex.direct.core.smsauth.storage.SmsAuthStorage;

import static ru.yandex.direct.core.smsauth.SmsAuthStorageUtils.getHash;

/**
 * Core сервис для SMS аутетификации.
 */
@Service
@ParametersAreNonnullByDefault
public class SmsAuthStorageService {

    private static final int MAX_PASSWORD_ATTEMPTS = 3;
    private static final int MAX_SMS_PER_DAY = 5;

    private final SmsAuthStorage smsAuthStorage;

    @Autowired
    public SmsAuthStorageService(SmsAuthStorage smsAuthStorage) {
        this.smsAuthStorage = smsAuthStorage;
    }

    /**
     * Проверяет залочена ли сессия и локает ее (атомарно) для ручек аутентификации через смс.
     *
     * @return true - если сессия уже залочена, иначе false
     */
    public boolean checkAndLockSession(SmsAuthKey key) {
        return smsAuthStorage.lockSession(key);
    }

    /**
     * Снимает лок с сессии для ручек аутентификации через смс.
     *
     * @return true - если сессия была залочена, иначе false
     */
    public boolean unlockSession(SmsAuthKey key) {
        return smsAuthStorage.unlockSession(key);
    }

    /**
     * Проверяет, было ли отправлено смс с паролем.
     *
     * @return true - если смс было отправлено, иначе false
     */
    public boolean checkSmsIsSent(SmsAuthKey key) {
        return smsAuthStorage.checkPasswordExists(key);
    }

    /**
     * Проверяет, может ли пользователь запросить еще одну смс с паролем.
     *
     * @param operatorUid   uid оператора
     * @return true - если пользователь исчерпал суточный лимит отправляемых смс, иначе false
     */
    public boolean checkUserCanSendSms(Long operatorUid) {
        String saltedOperatorUid = getHash(operatorUid.toString());

        return smsAuthStorage.incrementAndGetSentSmsCount(saltedOperatorUid) <= MAX_SMS_PER_DAY;
    }

    /**
     * Проверяет, может ли пользователь запросить еще одну смс с паролем.
     *
     * @return true - если пользователь исчерпал суточный лимит отправляемых смс, иначе false
     */
    public boolean userCanTryAgain(SmsAuthKey key) {
        return smsAuthStorage.incrementAndGetTriesCount(key) <= MAX_PASSWORD_ATTEMPTS;
    }

    /**
     * Проверяет прошел ли пользователь аутентификацию через смс.
     *
     * @return true - если пользователь прошел аутентификацию, иначе false
     */
    public boolean isUserAuthenticated(SmsAuthKey key) {
        return smsAuthStorage.getSuccess(key);
    }

    /**
     * Метод для проверки пароля из смс.
     *
     * @param sessionIdHash     хэш от id сессии
     * @param smsPassword       пароль, отправленный пользователем
     * @return true - если пароль, присланный пользователем, совпадает с отправленным в смс, иначе false
     */
    public boolean checkPassword(SmsAuthKey key, String sessionIdHash, String smsPassword) {
        String saltedPassword = getHash(smsPassword, sessionIdHash);

        return Objects.equals(smsAuthStorage.getPassword(key), saltedPassword);
    }

    /**
     * Метод для инициализации данных в Redis при отправке смс.
     *
     * @param sessionIdHash     хэш от id сессии
     * @param smsPassword       пароль, отправленный пользователем
     * @return true - если инициализация данных в Redis прошла успешно, иначе false
     */
    public boolean initAuthAttempt(SmsAuthKey key, String sessionIdHash, String smsPassword) {
        String saltedPassword = getHash(smsPassword, sessionIdHash);

        deleteAuthAttempt(key);

        return smsAuthStorage.setPassword(key, saltedPassword)
                && smsAuthStorage.setTriesCount(key);
    }

    /**
     * Метод для успешного завершения процесса аутентификации через смс.
     *
     * @return true - если инициализация данных в Redis прошла успешно, иначе false
     */
    public boolean completeAuthentication(SmsAuthKey key) {
        return deleteAuthAttempt(key)
                && smsAuthStorage.setSuccess(key);
    }

    /**
     * Метод для очистки данных из Redis.
     *
     * @return true - если инициализация данных в Redis прошла успешно, иначе false
     */
    private boolean deleteAuthAttempt(SmsAuthKey key) {
        // Здесь важно использовать именно bitwise and, чтобы очистка не остановилась на пол пути
        return smsAuthStorage.deletePassword(key)
                & smsAuthStorage.deleteTriesCount(key);
    }
}
