package ru.yandex.direct.core.smsauth.storage;

import java.time.Duration;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.redis.LettuceStorage;
import ru.yandex.direct.core.smsauth.model.SmsAuthKey;

import static java.lang.String.format;

/**
 * Storage для хранения и обработки полей, использующихся для аутентификации по смс, в Redis.
 */
@Component
@ParametersAreNonnullByDefault
public class SmsAuthStorage {

    private static final Logger logger = LoggerFactory.getLogger(SmsAuthStorage.class);

    private static final String KEY_FORMAT = "sms-auth:%s:%s";

    private static final String LOCK_FIELD = "lock";
    private static final String PASSWORD_FIELD = "password";
    private static final String TRIES_COUNT_FIELD = "tries-count";
    private static final String SENT_SMS_COUNT = "sent-sms-count";
    private static final String SUCCESS_FIELD = "success";

    private static final int SLIDING_WINDOW_DURATION_IN_SECONDS = (int) Duration.ofDays(1).toSeconds();
    private static final int SUCCESS_DURATION_IN_SECONDS = (int) Duration.ofMinutes(20).toSeconds();

    private static final int LOCK_SESSION_TIME_IN_SECONDS = 120;
    private static final int TIMER_IN_SECONDS = 180;
    private static final int TIMER_DELTA_IN_SECONDS = 30;
    private static final int TTL_IN_SECONDS = TIMER_IN_SECONDS + TIMER_DELTA_IN_SECONDS;

    private static final String LOCK_VALUE = "1";
    private static final String SUCCESS_VALUE = "1";

    private final LettuceStorage lettuceStorage;
    private final String keyPrefix;

    @Autowired
    public SmsAuthStorage(LettuceStorage lettuceStorage,
                          @Value("${sms_auth.key_prefix:}") String keyPrefix) {
        this.lettuceStorage = lettuceStorage;
        this.keyPrefix = keyPrefix;
    }

    public boolean setPassword(SmsAuthKey key, String saltedPassword) {
        String redisKey = getRedisKey(key, PASSWORD_FIELD);

        return lettuceStorage.setAndExpire(redisKey, saltedPassword, TTL_IN_SECONDS);
    }

    public String getPassword(SmsAuthKey key) {
        String redisKey = getRedisKey(key, PASSWORD_FIELD);

        return lettuceStorage.get(redisKey);
    }

    public boolean deletePassword(SmsAuthKey key) {
        String redisKey = getRedisKey(key, PASSWORD_FIELD);

        return lettuceStorage.delete(redisKey) == 1;
    }

    public boolean checkPasswordExists(SmsAuthKey key) {
        String redisKey = getRedisKey(key, PASSWORD_FIELD);

        return lettuceStorage.getTtl(redisKey) > TIMER_DELTA_IN_SECONDS;
    }

    public boolean setTriesCount(SmsAuthKey key) {
        String redisKey = getRedisKey(key, TRIES_COUNT_FIELD);

        return lettuceStorage.setAndExpire(redisKey, 0, TTL_IN_SECONDS);
    }

    public int incrementAndGetTriesCount(SmsAuthKey key) {
        String redisKey = getRedisKey(key, TRIES_COUNT_FIELD);

        return (int) lettuceStorage.incrementAndGet(redisKey);
    }

    public boolean deleteTriesCount(SmsAuthKey key) {
        String redisKey = getRedisKey(key, TRIES_COUNT_FIELD);

        return lettuceStorage.delete(redisKey) == 1;
    }

    public boolean lockSession(SmsAuthKey key) {
        String redisKey = getRedisKey(key, LOCK_FIELD);
        boolean lockIsSet = !LOCK_VALUE.equals(lettuceStorage.getAndSet(redisKey, LOCK_VALUE));

        // Вешаем expire на лок, чтобы в конечном итоге он отпустился, но вешаем только если удалось его повесить
        // внутри текущего запроса, чтобы не увеличить время лока другого запроса
        if (lockIsSet) {
            lettuceStorage.expire(redisKey, LOCK_SESSION_TIME_IN_SECONDS);
        }

        return lockIsSet;
    }

    public boolean unlockSession(SmsAuthKey key) {
        String redisKey = getRedisKey(key, LOCK_FIELD);

        return lettuceStorage.delete(redisKey) == 1;
    }

    public boolean setSuccess(SmsAuthKey key) {
        String redisKey = getRedisKey(key, SUCCESS_FIELD);

        return lettuceStorage.setAndExpire(redisKey, SUCCESS_VALUE, SUCCESS_DURATION_IN_SECONDS);
    }

    public boolean getSuccess(SmsAuthKey key) {
        String redisKey = getRedisKey(key, SUCCESS_FIELD);

        return SUCCESS_VALUE.equals(lettuceStorage.get(redisKey));
    }

    public int incrementAndGetSentSmsCount(String saltedOperatorUid) {
        String key = getKey(saltedOperatorUid, SENT_SMS_COUNT);
        long sentSmsCount = lettuceStorage.incrementAndGet(key);

        // Если новое значение 1 - значит этого ключа раньше не было. В этом случае применяем к нему expire, чтобы
        // ограничение на количество отправленных смс исчезло через какое-то время.
        if (sentSmsCount == 1) {
            lettuceStorage.expire(key, SLIDING_WINDOW_DURATION_IN_SECONDS);
        }

        return (int) sentSmsCount;
    }

    private String getKey(String saltedOperatorUid, String fieldName) {
        return keyPrefix + format(KEY_FORMAT, saltedOperatorUid, fieldName);
    }

    private String getRedisKey(SmsAuthKey key, String fieldName) {
        return keyPrefix + format(KEY_FORMAT, key.getSalted(), fieldName);
    }
}
