package ru.yandex.direct.core.units;

import java.util.Map;

import javax.annotation.Nullable;

import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static java.util.stream.Collectors.toMap;

public class Costs {
    static final String COST_MULTIPLIERS_CONFIG_KEY = "cost-multipliers";
    static final String OPERATIONS_COSTS_CONFIG_KEY = "costs";
    static final String DEFAULT_REQUEST_ERROR_COST_CONFIG_KEY = "common-request-error.default";
    static final String REQUEST_ERROR_COSTS_CONFIG_KEY = "common-request-error.codes";
    private static final double DEFAULT_APP_COEF = 1.0;
    private static final Logger logger = LoggerFactory.getLogger(Costs.class);

    private final Config serviceCostConfig;
    private final Config appCoefConfig;
    /**
     * Кол-во баллов, которое списывается в случае ошибки на уровне запроса
     * в случае если код ошибки не специфицирован или для заданного кода ошибки
     * кол-во баллов явно не заданно
     */
    private final Integer defaultRequestErrorCost;
    /**
     * Стоимость заданного типа ошибки в случае ошибки на уровне запроса,
     * т.е. отображение кода ошибки в стоимость в баллах
     */
    private final Map<Integer, Integer> requestErrorCosts;

    /**
     * Given {@code costsConfig} should contain next root-level paths:
     * <ul>
     * <li><code>costs</code> with costs for services in use</li>
     * <li><code>cost-multipliers</code> with application multiplier coefficients</li>
     * </ul>
     *
     * @param costsConfig config
     */
    public Costs(Config costsConfig) {
        // We don't want to store root config itself
        serviceCostConfig = costsConfig.getConfig(OPERATIONS_COSTS_CONFIG_KEY);
        appCoefConfig = costsConfig.getConfig(COST_MULTIPLIERS_CONFIG_KEY);
        defaultRequestErrorCost = costsConfig.getInt(DEFAULT_REQUEST_ERROR_COST_CONFIG_KEY);
        requestErrorCosts = costsConfig.getConfig(REQUEST_ERROR_COSTS_CONFIG_KEY)
                .entrySet()
                .stream()
                .collect(
                        toMap(
                                // Код ошибки
                                e -> Integer.valueOf(e.getKey()),
                                // Стоимость в баллах. Ожидаем Number (в силу того что конфиг наш,
                                // считаем что способны задать правильно)
                                e -> ((Number) e.getValue().unwrapped()).intValue()));
    }

    /**
     * @param applicationId {@link String} application ID
     * @return multiplier for requests costs related to given {@code applicationId}.
     * Default is {@link #DEFAULT_APP_COEF}
     */
    public double getApplicationCoef(String applicationId) {
        if (appCoefConfig.hasPath(applicationId)) {
            return appCoefConfig.getDouble(applicationId);
        }
        logger.debug("Can't find application coef for applicationId {} return default", applicationId);
        return DEFAULT_APP_COEF;
    }

    /**
     * Возвращает стоимость в баллах заданного типа ошибки
     *
     * @param errorCode Код ошибки или null (В случае null списывается default-ое кол-во баллов)
     * @return Кол-во баллов которое необходимо списать
     */
    public int getServiceErrorCost(@Nullable Integer errorCode) {
        return errorCode != null
                ? requestErrorCosts.getOrDefault(errorCode, defaultRequestErrorCost)
                : defaultRequestErrorCost;
    }

    /**
     * @return {@link OperationCosts} для метода, определяемого переданными параметрами {@code service} и
     * {@code operation}
     */
    public OperationCosts getOperationCosts(String service, String operation) {
        return new OperationCosts(serviceCostConfig.getConfig(service + "." + operation));
    }
}
