package ru.yandex.direct.core.units;

import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Позволяет получать стоимости вызова и выполнения конкретной операции.
 *
 * @see Costs
 */
public class OperationCosts {
    private static final Logger logger = LoggerFactory.getLogger(OperationCosts.class);

    public static final String CALL_COST_KEY = "call-cost";
    public static final String OBJECT_COST_KEY = "object-cost";
    public static final String SPECIAL_PROCESSING_OBJECT_COST_KEY = "special-processing-object-cost";
    public static final String MIN_DAILY_LIMIT_KEY = "min-daily-limit";
    public static final String OBJECT_ERROR_COST_KEY = "object-error-cost";
    public static final String REQUEST_ERROR_COST_KEY = "request-error-cost";
    public static final String SPENT_GROUP_SIZE_KEY = "spent-group-size";
    private static final int SINGLE_ELEMENT_GROUP_SIZE = 1;

    private final Config operationCosts;


    OperationCosts(Config operationCosts) {
        this.operationCosts = operationCosts;
    }

    /**
     * @return стоимость вызова
     */
    public int getCostForOperation() {
        return operationCosts.getInt(CALL_COST_KEY);
    }

    /**
     * @param count количество успешно обработанных объектов
     * @return стоимость выполнения операции над указанным числом объектов
     */
    private int getCostForObjects(int count) {
        return operationCosts.getInt(OBJECT_COST_KEY) * count;
    }

    /**
     * @return стоимость вызова при выполении особой операции требующей большего списание баллов
     * например, в методе {@code Keywords.get} за запрос со статистикой списывается бОльшее количество баллов API.
     */
    private int getSpecialProcessingObjectCost() {
        return operationCosts.getInt(SPECIAL_PROCESSING_OBJECT_COST_KEY);
    }

    /**
     * @param count количество успешно обработанных объектов
     * @return стоимость выполнения особой операции над указанным числом объектов
     * например, в методе {@code Keywords.get} за запрос со статистикой списывается бОльшее количество баллов API.
     */
    private int getSpecialProcessingObjectCost(int count) {
        return operationCosts.getInt(SPECIAL_PROCESSING_OBJECT_COST_KEY) * count;
    }

    /**
     * @param count количество объектов, которые не удалось обработать
     * @return стоимость ошибки операции для указанного числа объектов
     */
    private int getCostForObjectsWithErrors(int count) {
        return operationCosts.getInt(OBJECT_ERROR_COST_KEY) * count;
    }

    /**
     * @return стоимость ошибки валидации запроса
     */
    private int getCostForRequestError() {
        return operationCosts.getInt(REQUEST_ERROR_COST_KEY);
    }

    /**
     * @return какой минимальный размер корзины баллов должен быть у клиента, чтобы
     * у него был доступ к этому методу; если в конфигурации минимального размера корзины
     * для этого нет, возвращает 0: у всех клиентов положительный размер корзины, так
     * что всем можно
     */
    public int getMinDailyLimit() {
        if (!operationCosts.hasPath(MIN_DAILY_LIMIT_KEY)) {
            return 0;
        }

        return operationCosts.getInt(MIN_DAILY_LIMIT_KEY);
    }

    /**
     * Возвращает размер группы элементов, которые должны считаться за один объект для списания баллов.
     * Для некоторых методов вроде {@code Bids.get} баллы списываются не за каждый возвращённый элемент, а за группу.
     *
     * @return размер группы, за которую списываются баллы как за один объект.
     * Если в конфигурации минимального размера корзины для этого нет, возвращает {@value SINGLE_ELEMENT_GROUP_SIZE}
     */
    @SuppressWarnings("WeakerAccess")
    public int getSpentGroupSize() {
        if (!operationCosts.hasPath(SPENT_GROUP_SIZE_KEY)) {
            return SINGLE_ELEMENT_GROUP_SIZE;
        }

        return operationCosts.getInt(SPENT_GROUP_SIZE_KEY);
    }

    /**
     * На основе {@code fetchResult} вычисляет стоимость <b>всего</b> вызова, описываемого текущим
     * {@link OperationSummary}.
     *
     * @param summary описание результата выполнения операции
     * @return стоимость текущей выполненной операции
     */
    public int calcRequestCost(OperationSummary summary) {
        int spent = 0;
        spent += getCostForOperation();
        if (summary.isHasOperationError()) {
            spent += getCostForRequestError();
        } else {
            int objectsSuccessCount = summary.getObjectsSuccessCount();
            // Для большинства операций размер группы равен единице, что означает списание баллов за каждый
            int spentGroupCount = objectsSuccessCount / getSpentGroupSize();
            if (summary.isSpecialProcessingObjectCost()) {
                spent += getSpecialProcessingObjectCost(spentGroupCount);
            } else {
                spent += getCostForObjects(spentGroupCount);
            }
            spent += getCostForObjectsWithErrors(summary.getObjectsErrorCount());
        }
        logger.debug("Cost of {}: {}", summary, spent);
        return spent;
    }

    @Override
    public String toString() {
        return "OperationCosts{" +
                "operationCosts=" + operationCosts +
                '}';
    }
}
