package ru.yandex.direct.core.units;

import com.google.common.base.Objects;

/**
 * Класс-контейнер, хранящий общую информацию о текущей операции.
 *
 * @see OperationCosts#calcRequestCost(OperationSummary)
 */
public class OperationSummary {
    private final boolean hasOperationError;
    private final int objectsSuccessCount;
    private final int objectsErrorCount;
    private final boolean specialProcessingObjectCost;

    private OperationSummary(boolean hasOperationError, int objectsSuccessCount, int objectsErrorCount,
                             boolean specialProcessingObjectCost) {
        this.hasOperationError = hasOperationError;
        this.objectsErrorCount = objectsErrorCount;
        this.objectsSuccessCount = objectsSuccessCount;
        this.specialProcessingObjectCost = specialProcessingObjectCost;
    }

    /**
     * @return {@link OperationSummary}, соответствующей успешной операции, обобщённая статистика выполнения которой
     * описывается переданными параметрами
     */
    public static OperationSummary successful(int objectsSuccessCount, int objectsErrorCount) {
        return new OperationSummary(false, objectsSuccessCount, objectsErrorCount, false);
    }

    public static OperationSummary successful(int objectsSuccessCount) {
        return new OperationSummary(false, objectsSuccessCount, 0, false);
    }

    public static OperationSummary successful(int objectsSuccessCount, boolean specialProcessingObjectCost) {
        return new OperationSummary(false, objectsSuccessCount, 0, specialProcessingObjectCost);
    }

    /**
     * @return {@link OperationSummary}, соответствующей неуспешной операции. В таком случае нас нет информации о
     * количестве объектов в запросе
     */
    public static OperationSummary unsuccessful() {
        return new OperationSummary(true, 0, 0, false);
    }


    boolean isHasOperationError() {
        return hasOperationError;
    }

    int getObjectsErrorCount() {
        return objectsErrorCount;
    }

    public int getObjectsSuccessCount() {
        return objectsSuccessCount;
    }

    /**
     * Данный флаг может быть использован, когда у метода может быть несколько вариантов стоимости объекта.
     * Например, в методе {@code Keywords.get} за запрос со статистикой списывается бОльшее количество баллов API.
     */
    boolean isSpecialProcessingObjectCost() {
        return specialProcessingObjectCost;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        OperationSummary that = (OperationSummary) o;
        return hasOperationError == that.hasOperationError &&
                objectsSuccessCount == that.objectsSuccessCount &&
                objectsErrorCount == that.objectsErrorCount &&
                specialProcessingObjectCost == that.specialProcessingObjectCost;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(hasOperationError, objectsSuccessCount, objectsErrorCount, specialProcessingObjectCost);
    }

    @Override
    public String toString() {
        return "OperationSummary{" +
                "hasOperationError=" + hasOperationError +
                ", objectsErrorCount=" + objectsErrorCount +
                ", objectsSuccessCount=" + objectsSuccessCount +
                ", specialProcessingObjectCost=" + specialProcessingObjectCost +
                '}';
    }
}
