package ru.yandex.direct.core.units.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.units.api.UnitsBalance;
import ru.yandex.direct.core.units.api.UnitsBalanceImpl;
import ru.yandex.direct.core.units.storage.LettuceStorage;
import ru.yandex.direct.core.units.storage.Storage;

/**
 * Концепция баллов введена для ограничения количества и сложности запросов
 * от клиентов. Каждой операции поставлено в соответствие определенное количество
 * баллов (в зависимости от количества необходимых для ее выполнения ресурсов
 * Direct-а), которое будет списано при ее выполнении. Каждому клиенту назначается
 * определённое количество баллов, которые он может израсходовать в рамках
 * скользящего окна, выполняя какие-либо операции. Скользящее окно - это набор
 * последовательных временных интервалов равной длины. Интервал является
 * минимальной единицей измерения времени, за которое учитываются израсходованные
 * клиентом баллы. При достижении максимально разрешенного количества баллов в
 * рамках такого скользящего окна клиенту разрешается делать только некоторые
 * типы запросов.
 * <p>
 * Сервис преднзначен для получения из БД информации о текущем состоянии баланса баллов пользователя
 * ({@link #getUnitsBalance(ApiUser)}) и для сохранения информации о вновь списываемых баллах
 * ({@link #updateSpent(UnitsBalance)}).
 */
@Service
public class UnitsService {
    public static final int DEFAULT_LIMIT = 160_000;
    public static final int DEFAULT_INTERVALS = 24;
    public static final String DEFAULT_BASKET_NAME = "API";
    public static final int DEFAULT_INTERVAL_SIZE_SEC = (int) TimeUnit.HOURS.toSeconds(1); //SECONDS_PER_HOUR
    /**
     * Так как сейчас основная имплементация {@link Storage} &ndash; {@link ru.yandex.direct.core.units.storage.LettuceStorage},
     * тут применена оптимизация для Redis: ClientId упаковывается в фигурные скобочки, чтобы все значени для одного клиента
     * хранились на одной ноде Redis-кластера.
     * Для определения ноды используется hash-значение, вычисленное от подстроки в первых фигурных скобочках.
     * Подробнее: <a href="http://redis.io/topics/cluster-spec#keys-hash-tags">Cluster spec: Keys hash tags</a>
     * <p>
     * В данную заготовку передаются следующие параметры:
     * <ol>
     * <li><code>clientId</code>;</li>
     * <li><code>basket name</code> - например, 'API';</li>
     * <li><code>interval</code> - координаты временного интервала;</li>
     * </ol>
     */
    public static final String SPENT_UNITS_KEY_FORMAT = "units-{%d}-%s-%d-spent";
    private static final Logger logger = LoggerFactory.getLogger(UnitsService.class);
    private final Storage storage;
    private final String keyPrefix;

    @Autowired
    public UnitsService(@Qualifier(LettuceStorage.NAME) Storage storage,
                        @Value("${units.key_prefix:}") String keyPrefix) {
        this.storage = storage;
        this.keyPrefix = keyPrefix;
    }

    public UnitsService(Storage storage) {
        this(storage, "");
    }

    private UnitsBalance createUnitsBalance(ApiUser unitsHolder, int balance, int spent) {
        return new UnitsBalanceImpl(
                unitsHolder.getClientId().asLong(),
                getLimit(unitsHolder),
                balance,
                spent);
    }

    /**
     * Создать фиктивный баланс баллов
     * <p>
     * Используется в случае недоступности хранилища баллов
     */
    public UnitsBalance createFallbackUnitsBalance(ApiUser unitsHolder) {
        int limit = getLimit(unitsHolder);
        return new UnitsBalanceImpl(
                unitsHolder.getClientId().asLong(),
                limit,
                limit,
                // Мы не знаем сколько баллов реально потраченно, поэтому считаем что доступен весь суточный лимит
                0);
    }

    /**
     * Записывает в БД информацию о списанных в рамках текущей операции баллах.
     *
     * @param unitsBalance {@link UnitsBalance}, содержащий информацию о том, сколько баллов было списано
     * @return {@code int} &ndash; число реально списанных баллов. {@code 0}, если списание не удалось.
     */
    public int updateSpent(UnitsBalance unitsBalance) {
        int spentInCurrentRequest = unitsBalance.spentInCurrentRequest();
        logger.debug("updateSpent({})", spentInCurrentRequest);
        if (spentInCurrentRequest == 0) {
            logger.debug("Nothing spent. Skip storage call");
            return 0;
        }

        String key = getIntervalKey(unitsBalance.getClientId(), getCurrentInterval());
        if (storage.incrOrSet(key, spentInCurrentRequest, getTimeToLive())) {
            return spentInCurrentRequest;
        }

        logger.warn("Can't updateSpent {} units for clientId {}", spentInCurrentRequest, unitsBalance.getClientId());
        return 0;
    }

    /**
     * Вычисляет текущий баланс и количество израсходованных баллов для указанного пользователя
     */
    public UnitsBalance getUnitsBalance(ApiUser unitsHolder) {
        // Получаем потраченные баллы за последние 24 часа по интервалам (сейчас часам), включая текущий интервал (сейчас час)
        Map<Interval, Integer> spentByIntervals = getSpentByIntervals(unitsHolder);
        int balance = calcBalance(unitsHolder, spentByIntervals);
        int spent = calcSpent(spentByIntervals);
        return createUnitsBalance(unitsHolder, balance, spent);
    }

    public int getLimit(ApiUser unitsHolder) {
        // повторяем логику perl-api, null и 0 - означает баллы не заданы, нужно использовать дефолт
        long limit = Math.max(
                Optional.ofNullable(unitsHolder.getApiUnitsDaily()).orElse(0L),
                Optional.ofNullable(unitsHolder.getApi5UnitsDaily()).orElse(0L)
        );
        if (limit != 0) {
            return (int) limit;
        } else {
            return DEFAULT_LIMIT;
        }
    }

    private int calcBalance(ApiUser unitsHolder, Map<Interval, Integer> spentByIntervals) {
        int limit = getLimit(unitsHolder);
        double intervalLimit = 1. * limit / DEFAULT_INTERVALS;

        // Доступное нам количество баллов равно кол-ву баллов за предшестующие
        // 23 часа (если что-то осталось) + 1/24 от суточного лимита. Для того, чтобы
        // гарантировать это мы должны отсортировать интервалы
        // Подробности здесь https://tech.yandex.ru/direct/doc/dg/concepts/units-docpage/
        List<Interval> keysSorted = spentByIntervals.keySet().stream()
                .sorted().collect(Collectors.toList());
        double balanceDouble = 0.;
        for (Interval interval : keysSorted) {
            int spent = spentByIntervals.get(interval);
            /*
             * Если частичная сумма уходит в минус, значит были потрачены баллы, накопленные за пределами текущего окна
             * Такой долг "забываем"
             */
            balanceDouble = Math.max(0., balanceDouble + (intervalLimit - spent));
        }
        return (int) Math.round(balanceDouble);
    }

    private int calcSpent(Map<Interval, Integer> spentByIntervals) {
        return spentByIntervals.values()
                .stream()
                .mapToInt(Integer::valueOf)
                .sum();
    }

    /**
     * @param unitsHolder {@link ApiUser}
     * @return {@link Map}&lt;{@link Interval}, {@link Integer}&gt; где для каждого интервала хранится количество
     * потраченных баллов. Если в БД не нашлось записи, для такого интервала вернётся 0, а не {@code null}.
     */
    private Map<Interval, Integer> getSpentByIntervals(ApiUser unitsHolder) {
        List<Interval> intervals = getSlidingWindowIntervals(DEFAULT_INTERVALS);

        Map<String, Interval> intervalsByKeys =
                intervals.stream()
                        .collect(Collectors.toMap(i -> getIntervalKey(unitsHolder.getClientId().asLong(), i), i -> i));
        Map<String, Integer> spentByKeys = storage.getMulti(intervalsByKeys.keySet());

        return intervalsByKeys.keySet()
                .stream()
                .collect(Collectors.toMap(intervalsByKeys::get,
                        key -> Optional.ofNullable(spentByKeys.get(key)).orElse(0)));
    }

    private List<Interval> getSlidingWindowIntervals(int intervalsAmount) {
        Interval current = getCurrentInterval();
        return IntStream.range(0, intervalsAmount)
                .mapToObj(current::prevBy)
                .collect(Collectors.toList());
    }


    private Interval getCurrentInterval() {
        return new Interval(getEpochSeconds(), DEFAULT_INTERVAL_SIZE_SEC);
    }

    protected long getEpochSeconds() {
        return (System.currentTimeMillis() / 1000L);
    }

    /**
     * Return key by given interval
     *
     * @param interval interval
     * @return {@link String} key
     */
    private String getIntervalKey(Long clientId, Interval interval) {
        return keyPrefix + String.format(SPENT_UNITS_KEY_FORMAT, clientId, DEFAULT_BASKET_NAME, interval.value());
    }

    private int getTimeToLive() {
        return DEFAULT_INTERVALS * DEFAULT_INTERVAL_SIZE_SEC;
    }

    /*
      Следующие методы требуются лишь для корреткировки объёма потраченных баллов во время интеграционных тестов
     */
    //fixme: если этот метод не будет использоваться в интеграционных тестах, его необходимо удалить
    protected boolean restoreBalance(ApiUser unitsHolder) {
        List<String> keys = getSlidingWindowKeys(unitsHolder);
        Map<String, Integer> deleted = storage.deleteMulti(keys);
        return keys.stream()
                .filter(deleted::containsKey)
                .count() == keys.size();
    }

    private List<String> getSlidingWindowKeys(ApiUser unitsHolder) {
        List<Interval> intervals = getSlidingWindowIntervals(DEFAULT_INTERVALS);
        return intervals.stream()
                .map(i -> getIntervalKey(unitsHolder.getClientId().asLong(), i))
                .collect(Collectors.toList());
    }

    /**
     * Класс отображает временной интервал, для которого в БД хранится информация о потраченных баллах.
     * Помимо типизации класс также предоставляет механизм получения N-ого предыдущего интервала - {@link #prevBy(int)},
     * что необходимо при вычислении баланса пользователя на текущий момент.
     */
    private static class Interval implements Comparable<Interval> {
        private final long interval;
        private final int intervalSizeSec;

        Interval(long epochTime, int intervalSizeSec) {
            // Округляем значение вниз до ближайшего кратного intervalSizeSec
            this.interval = (epochTime / intervalSizeSec) * intervalSizeSec;
            this.intervalSizeSec = intervalSizeSec;
        }

        long value() {
            return interval;
        }

        /**
         * Позволяет получить {@code relPos} предыдущий интервал.
         *
         * @param relPos параметр, задающий координаты искомого интервала
         * @return {@link Interval}, который отстоит от текущего на {@code relPos} позиций
         */
        Interval prevBy(int relPos) {
            return new Interval(interval - relPos * intervalSizeSec, intervalSizeSec);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            Interval interval1 = (Interval) o;

            return interval == interval1.interval;

        }

        @Override
        public int hashCode() {
            return (int) (interval ^ (interval >>> 32));
        }

        @Override
        public int compareTo(@Nonnull Interval o) {
            return Long.compare(interval, o.interval);
        }
    }
}
