package ru.yandex.direct.core.units.storage;

import java.util.Collection;
import java.util.Map;

/**
 * Определяет интерфейс, который backend хранилища баллов должен реализовывать
 */
public interface Storage {

    /**
     * Получает из хранилища значение по ключу
     *
     * @param key ключ
     * @return целочисленное значение
     */
    Integer get(String key);

    /**
     * Получает из хранилища несколько значений по указанным ключам
     *
     * @param keys {@link Collection}&lt;{@link String}&gt; с набором ключей, которые необходимо получить
     * @return {@link Map}&lt;{@link String}, {@link Integer}&gt; хранящяя значения для найденных ключей
     */
    Map<String, Integer> getMulti(Collection<String> keys);

    /**
     * Устанавливает в хранилище указанное значение по указанному ключу с указанным временем time-to-live
     *
     * @param key   {@link String} key
     * @param value {@link String} keyvalue to set
     * @param ttl   Time-To-Live
     * @return {@code true} if success {@code false} otherwise
     */
    boolean set(String key, Integer value, int ttl);

    /**
     * Увеличивает в хранилище значение по указанному ключу на указанную величину
     *
     * @param key   {@link String} key
     * @param delta {@link Integer} amount to be added to stored value
     * @return {@code true} if success {@code false} otherwise
     */
    boolean incr(String key, Integer delta);

    /**
     * If value stored by {@code key} is present then it is incremented by {@code delta}. Otherwise value is stored to DB with defined {@code ttl}
     *
     * @param key   key
     * @param delta value to be added or store
     * @param ttl   Time To Live
     * @return {@code true} if success {@code false} otherwise
     */
    default boolean incrOrSet(String key, Integer delta, int ttl) {
        return incr(key, delta) || set(key, delta, ttl);
    }

    /**
     * Удаляет из хранилища значения по указанным ключам
     *
     * @param keys {@link Collection}&lt;{@link String}&gt; keys to be deleted
     * @return {@link Map}&lt;{@link String}, {@link Integer}&gt; with values for deleted keys being found
     */
    Map<String, Integer> deleteMulti(Collection<String> keys);

}
