package ru.yandex.direct.core.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Для переданного в конструктор geoTree строит вспомогательные структуры для работы над деревом.
 * Если нужна какая-то операция над деревом geoTree - она должна быть здесь в виде метода.
 */
@ParametersAreNonnullByDefault
public class GeoTreeConverter {
    private final Map<Long, Region> regions;
    private final Map<Long, List<Long>> regionsChildren;

    public GeoTreeConverter(GeoTree geoTree) {
        this.regions = geoTree.getRegions();
        this.regionsChildren = buildRegionsChildren(regions);
    }

    private static Map<Long, List<Long>> buildRegionsChildren(Map<Long, Region> regions) {
        var regionsChildren = new HashMap<Long, List<Long>>();
        regions.values().forEach(region -> {
            long regionId = region.getId();
            regionsChildren.putIfAbsent(regionId, new ArrayList<>());

            if (region.getParent() == null) {
                return;
            }

            long parentRegionId = region.getParent().getId();
            // глобальный регион сам является своим родителем, игнорируем этот странный факт
            if (regionId == parentRegionId) {
                return;
            }

            regionsChildren.computeIfAbsent(parentRegionId, ignore -> new ArrayList<>())
                    .add(regionId);
        });
        return regionsChildren;
    }

    /**
     * "Развернуть" гео до указанного типа.
     * <p>
     * Принимает гео с регионами разного уровня (в том числе с минус-регионами)
     * и приводит его к списку гео с указанным типом.
     * <p>
     * Например: geo=[Россия, -Центр, Владивосток]; geoType=Регион
     * Результат: Все регионы России без регионов принадлежащих центральному округу.
     * (г. Владивосток не попадёт в список, т.к. его geoType=Город "мельче" запрошенного geoType=Регион)
     * <p>
     * Следует заметить, что операция не транзитивна. В дереве могут быть пропуски geoType. geoType может идти
     * от parent к children например так 5-4-2-1 (пропуская geoType 3). Если мы сделаем expandGeo сначала с 5 до 3,
     * а потом результат с 3 до 1 - это не то же самое, что сделать expandGeo с 5 сразу на 1.
     *
     * @param geo     гео, который надо развернуть
     * @param geoType до какого уровня развернуть
     * @return список гео
     */
    public List<Long> expandGeo(Collection<Long> geo, Integer geoType) {
        var result = new HashSet<Long>();
        geo.stream()
                .filter(regionId -> regionId >= 0)
                .forEach(regionId -> result.addAll(getChildRegionsOfType(regionId, geoType)));
        geo.stream()
                .filter(regionId -> regionId < 0)
                .forEach(regionId -> result.removeAll(getChildRegionsOfType(-regionId, geoType)));

        return new ArrayList<>(result);
    }

    /**
     * Получить дочерние регионы с указанным типом.
     * <p>
     * - Если тип parentRegion < переданного type на несколько уровней,
     *   то в результате будут только регионы указанного уровня, но не регионы промежуточных уровней
     *   Например: если тип parentRegion=страна(3), а type=регион(5), то в результат попадут только регионы,
     *   округа(4) - не попадут
     * - Если тип parentRegion = переданному type, то сам parentRegion попадёт в результат
     * - Если тип parentRegion > переданного type, то вернётся пустое множество
     *
     * @param parentRegionId регион, для которого ищем дочерние
     * @param regionType     тип искомых дочерних регионов
     * @return дочерние регионы указанного типа
     */
    private List<Long> getChildRegionsOfType(long parentRegionId, Integer regionType) {
        var result = new ArrayList<Long>();
        getChildRegionsOfType(parentRegionId, regionType, result);
        return result;
    }

    /**
     * Рекурсивная часть метода List<Long> getChildRegionsOfType.
     * Использование в других местах - не предполагается.
     */
    private void getChildRegionsOfType(long parentRegionId, Integer regionType, List<Long> result) {
        var parentRegion = regions.get(parentRegionId);
        checkNotNull(parentRegion, "Region not found");

        if (parentRegion.getType() == regionType) {
            result.add(parentRegionId);
        } else if (parentRegion.getType() < regionType) {
            // продолжаем поиск в дочерних элементах, только если текущий регион может содержать регионы искомого типа
            // например старана(type=3) может содержать округ(type=4), но не наоборот
            regionsChildren.get(parentRegionId)
                    .forEach(childRegionId -> getChildRegionsOfType(childRegionId, regionType, result));
        }
    }

    /**
     * На основе geoTree, который передан в конструктор этого класса строит дерево с корнем rootRegionId
     * В построенном дереве для листов в children будет лежат пустая коллекция (а не null).
     *
     * @param rootRegionId       id корня дерева
     * @param nodeConstructor    констуктор узла
     * @param nodeChildrenSetter функция связывающая узел с потомками
     * @param <T>                тип узла дерева
     * @return корень дерева
     */
    public <T> T buildRegionTree(Long rootRegionId,
                                 Function<Long, T> nodeConstructor,
                                 BiConsumer<T, List<T>> nodeChildrenSetter) {
        return TreeUtils.convertTree(rootRegionId, regionsChildren::get, nodeConstructor, nodeChildrenSetter);
    }

}
