package ru.yandex.direct.core.util;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.Region;

import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;

@ParametersAreNonnullByDefault
public class GeoTreeUtils {

    /**
     * Возвращает true, если каждый регион из regions1 не null и находится в гео-дереве, заданом в regions2
     * Например, "Москва, Санкт-Петербург" является поддеревом "Россия".
     * Переданные регионы должны существовать.
     * Игнорирует минус-регионы.
     */
    public static boolean isGeoSubTree(GeoTree geoTree, Collection<Long> regions1, Set<Long> regions2) {
        Set<Long> plusRegions1 = filterPlusRegions(regions1);
        Set<Long> plusRegions2 = filterPlusRegions(regions2);
        return plusRegions1.stream().allMatch(
                regionId -> regionId != null
                        && isGeoSubRegion(geoTree.getRegion(regionId), plusRegions2));
    }

    private static Set<Long> filterPlusRegions(Collection<Long> regions) {
        return filterToSet(regions, region -> region > 0L);
    }

    private static Set<Long> filterMinusRegions(Collection<Long> regions) {
        return filterToSet(regions, region -> region < 0L);
    }

    private static boolean isGeoSubRegion(Region region, Set<Long> regions2) {
        // глобальный регион сам является своим родителем, игнорируем этот странный факт
        boolean hasParent = region.getId() != region.getParent().getId();
        return regions2.contains(region.getId())
                || (hasParent && isGeoSubRegion(region.getParent(), regions2));
    }

    /**
     * Возвращает пересекаются ли два гео-дерева хотя бы по одному региону.
     * Минус-регионы допускаются в первом гео-дереве, но не допускаются во втором.
     * Например, Москва и Россия пересекаются в Москве.
     * А Москва и Санкт-Петербург - нигде не пересекаются
     *
     * @param geoTree    - полное гео-дерево
     * @param regionIds1 - гео-дерево, допускаются минус-регионы
     * @param regionIds2 - гео-дерево, не допускаются минус-регионы
     * @return           - true если деревья пересекаются хотя бы по одному региону, иначе false
     */
    public static boolean areTreesOverlap(GeoTree geoTree, Collection<Long> regionIds1, Set<Long> regionIds2) {
        Set<Long> plusRegions1 = filterPlusRegions(regionIds1);
        Set<Long> minusRegions1 = filterMinusRegions(regionIds1);
        return regionIds2.stream()
                .anyMatch(regionId -> areTreesOverlap(geoTree, plusRegions1, minusRegions1, regionId));
    }

    private static boolean areTreesOverlap(GeoTree geoTree,
                                           Set<Long> plusRegions,
                                           Set<Long> minusRegions,
                                           Long regionId) {
        var regionIdWithParents = addParents(geoTree, List.of(regionId));
        // если заминусован любой из родителей региона или сам регион - он не пересекается с деревом
        if (regionIdWithParents.stream().anyMatch(region -> minusRegions.contains(-region))) {
            return false;
        }
        // в противном случае минус регионы не играют роли и их можно больше не рассматривать
        // пересечение будет, если либо дерево содержит какого-то из парентов региона либо же
        // сам регион является parent какого-то эл-та дерева
        // Считаем, что регион минусуется полностью и потом нельзя какую-то часть его отплюсовать.
        // +Россия -Москва можно
        // -Россия +Москва нельзя
        var plusRegionsWithParents = addParents(geoTree, plusRegions);
        return regionIdWithParents.stream().anyMatch(plusRegions::contains)
                || plusRegionsWithParents.contains(regionId);
    }

    private static Set<Long> addParents(GeoTree geoTree, Collection<Long> regionIds) {
        Set<Long> parents = getParents(geoTree, regionIds);
        Set<Long> result = new HashSet<>(parents);
        result.addAll(regionIds);
        return result;
    }

    private static Set<Long> getParents(GeoTree geoTree, Collection<Long> regionIds) {
        Set<Long> result = new HashSet<>();
        regionIds.forEach(regionId -> fillParents(geoTree.getRegion(regionId), result));
        return result;
    }

    private static void fillParents(Region region, Set<Long> result) {
        while (parentExists(region)) {
            result.add(region.getParent().getId());
            region = region.getParent();
        }
    }

    private static boolean parentExists(Region region) {
        return region.getParent() != null && region.getParent().getId() != region.getId();
    }

}
