package ru.yandex.direct.core.util;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.bundle.TranslationBundle;

/**
 * Сервис для переводов по интерфейсу {@link TranslationBundle}
 *
 * Обходит с помощью reflection {@link TranslationBundle} дергает нужный метод по ключу и получает его
 * {@link Translatable} для перевода
 */
@Service
@ParametersAreNonnullByDefault
public class ReflectionTranslator {
    private final TranslationService translationService;
    private Cache<String, Translatable> translatableCache;

    private static final Logger logger = LoggerFactory.getLogger(ReflectionTranslator.class);

    public ReflectionTranslator(TranslationService translationService) {
        this.translationService = translationService;

        this.translatableCache = CacheBuilder.newBuilder()
                .maximumSize(200)
                .expireAfterWrite(1, TimeUnit.DAYS)
                .build();
    }

    public String translate(@Nullable String key, TranslationBundle translationBundle) {
        return getTranslation(key, translationBundle, translationService::translate);
    }

    public String translate(@Nullable String key, TranslationBundle translationBundle, Locale locale) {
        return getTranslation(
                key,
                translationBundle,
                translatable -> translationService.translate(translatable, locale)
        );
    }

    private String getTranslation(@Nullable String key,
                                  TranslationBundle translationBundle,
                                  Function<Translatable, String> translateFunction) {
        if (key == null) {
            return null;
        }

        // Включаем полный путь до класса с его именем в ключ кэша, потому что одинаковые ключи могут быть в разных
        // бандлах
        var cacheKey = translationBundle.getClass().getCanonicalName() + key;
        Translatable translatable = translatableCache.getIfPresent(cacheKey);
        if (translatable != null) {
            return translateFunction.apply(translatable);
        }
        for (Method method : translationBundle.getClass().getMethods()) {
            if (method.getName().equalsIgnoreCase(key)) {
                try {
                    translatable = (Translatable) method.invoke(translationBundle);
                    translatableCache.put(cacheKey, translatable);
                    return translateFunction.apply(translatable);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    logger.error("translateCategoryTypeDynamic reflection error", e);
                }
            }
        }
        return key;
    }
}
