package ru.yandex.direct.core.util;

import java.util.Collection;
import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public class TreeUtils {

    /**
     * Трансформирует одно представление дерева в другое.
     * Использовать только на валидных данных, поскольку реализовано рекурсивно без каких либо проверок на глубину
     * рекурсии.
     * В исходном дереве для листов в children может быть как null, так и пустая коллекция.
     *
     * @param sourceRoot           - root исходного дерева
     * @param sourceChildrenGetter - функция отдающая для исходного дерева children для узла
     * @param destNodeConstructor  - конструктор узла нового дерева
     * @param destChildrenSetter   - Консьюмер связывающий в новом дереве узел с его children.
     *                               Получает сконвертированные элемент и его сконвертированных children,
     *                               причём children приходят в том же порядке, что и в изначальном дереве.
     *                               Здесь можно произвести сортировку children, или проставить null для листов,
     *                               если это требуется.
     * @param <T>                  - тип узла нового дерева
     * @param <E>                  - тип узла исходного дерева
     * @return - root нового дерева
     */
    public static <T, E> T convertTree(E sourceRoot,
                                       Function<E, Collection<E>> sourceChildrenGetter,
                                       Function<E, T> destNodeConstructor,
                                       BiConsumer<T, List<T>> destChildrenSetter) {
        T destRoot = destNodeConstructor.apply(sourceRoot);
        Collection<E> sourceChildren = sourceChildrenGetter.apply(sourceRoot);
        List<T> destChildren = Stream.ofNullable(sourceChildren)
                .flatMap(Collection::stream)
                .map(sourceChild ->
                        convertTree(sourceChild, sourceChildrenGetter, destNodeConstructor, destChildrenSetter))
                .collect(Collectors.toList());
        destChildrenSetter.accept(destRoot, destChildren);
        return destRoot;
    }

}
