package ru.yandex.direct.core.validation;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.validation.defects.RightsDefects.forbiddenToChange;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.validation.result.PathHelper.field;

public final class ValidationUtils {

    private ValidationUtils() {
    }

    public static boolean hasValidationIssues(Result<?> result) {
        return result.getValidationResult() != null && hasValidationIssues(result.getValidationResult());
    }

    public static boolean hasValidationIssues(@Nullable ValidationResult validationResult) {
        return validationResult != null && validationResult.hasAnyErrors();
    }

    public static boolean hasAnyErrorsOrWarnings(@Nullable ValidationResult validationResult) {
        return validationResult != null
                && (validationResult.hasAnyErrors() || validationResult.hasAnyWarnings());
    }

    public static <T extends ModelWithId> void addErrorIfPropertyChanged(ModelChanges<T> mc,
                                                                         ItemValidationBuilder<ModelChanges<T>,
                                                                                 Defect> vb,
                                                                         ModelProperty property) {
        addErrorIfPropertyChanged(mc, vb, property, forbiddenToChange());
    }

    public static <T extends ModelWithId> void addErrorIfPropertyChanged(ModelChanges<T> mc,
                                                                         ItemValidationBuilder<ModelChanges<T>,
                                                                                 Defect> vb,
                                                                         ModelProperty property,
                                                                         Defect error) {
        if (mc.isPropChanged(property)) {
            vb.getResult()
                    .getOrCreateSubValidationResult(field(property), mc.getChangedProp(property))
                    .addError(error);
        }
    }

    public static <T extends ModelWithId> void addErrorIfPropertyChanged(AppliedChanges<T> ac,
                                                                         ItemValidationBuilder<T, Defect> vb,
                                                                         ModelProperty property) {
        addErrorIfPropertyChanged(ac, vb, property, forbiddenToChange());
    }

    public static <T extends ModelWithId> void addErrorIfPropertyChanged(AppliedChanges<T> ac,
                                                                         ItemValidationBuilder<T, Defect> vb,
                                                                         ModelProperty property,
                                                                         Defect error) {
        if (ac.changed(property)) {
            vb.getResult()
                    .getOrCreateSubValidationResult(field(property), ac.getNewValue(property))
                    .addError(error);
        }
    }

    public static <T extends ModelWithId> List<ModelProperty<? super T, ?>> getActuallyChangedProperties(T model,
            ModelChanges<T> modelChanges,
            Collection<ModelProperty<? super T, ?>> modelPropertiesForCheck) {

        return filterList(modelPropertiesForCheck, modelProperty -> {
            if (modelChanges.isPropChanged(modelProperty)) {
                var newValue = modelChanges.getPropIfChanged(modelProperty);
                return !areModelPropertyValuesEqual(modelProperty.get(model), newValue);
            }
            return false;
        });
    }

    public static <T extends ModelWithId> List<ModelProperty<? super T, ?>> getActuallyChangedProperties(T model,
            ModelChanges<T> modelChanges) {
        return getActuallyChangedProperties(model, modelChanges, modelChanges.getChangedPropsNames());
    }

    public static boolean areModelPropertyValuesEqual(@Nullable Object firstObject, @Nullable Object secondObject) {
        if (firstObject instanceof BigDecimal && secondObject instanceof BigDecimal) {
            return ((BigDecimal)firstObject).compareTo((BigDecimal)secondObject) == 0;
        }
        return Objects.equals(firstObject, secondObject);
    }
}
