package ru.yandex.direct.core.validation.constraints;

import java.math.BigDecimal;
import java.util.Set;
import java.util.regex.Pattern;

import com.google.common.net.InternetDomainName;

import ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants;
import ru.yandex.direct.core.validation.defects.MoneyDefects;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.constraint.AdmissibleCharsConstraint;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicateAndDefectSupplier;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.StringDefects.mustContainLettersOrDigitsOrPunctuations;

public class Constraints {
    private static final AdmissibleCharsConstraint ALLOWED_BANNER_LETTERS_CONSTRAINT =
            new AdmissibleCharsConstraint(BannerLettersConstants.ALLOW_BANNER_LETTERS_STR,
                    mustContainLettersOrDigitsOrPunctuations());

    //Это сильно урезанная проверка на логин, поверхностная слабая валидация, которая пропускает устаревшие и ПДД логины
    private static final Pattern LOGIN_PATTERN = Pattern.compile("^[a-zA-Z0-9-_.@]+$");

    private Constraints() {
    }

    // strings

    public static Constraint<String, Defect> allowedBannerLetters() {
        return ALLOWED_BANNER_LETTERS_CONSTRAINT;
    }


    public static Constraint<ClientId, Defect> validClientId() {
        return fromPredicate(not(clientId -> clientId.asLong() < 1L), CommonDefects.validId());
    }

    /**
     * Проверить корректность логина пользователя в приведенном формате
     */
    public static Constraint<String, Defect> validLogin() {
        return fromPredicate(v -> LOGIN_PATTERN.matcher(v).matches(), CommonDefects.validLogin());
    }

    /**
     * Проверка валидности домена
     */
    public static Constraint<String, Defect> validDomain() {
        return fromPredicate(domain -> {
            try {
                return InternetDomainName.from(domain).parts().size() > 1 && !domain.contains("_");
            } catch (IllegalArgumentException e) {
                return false;
            }
        }, invalidValue());
    }

    public static Constraint<BigDecimal, Defect> cpmNotLessThan(BigDecimal minPrice, CurrencyCode code) {
        // Money.valueOf кинет NPE на null, поэтому используем версию метода с Supplier
        return fromPredicateAndDefectSupplier(v -> v.compareTo(minPrice) >= 0,
                () -> MoneyDefects.invalidValueCpmNotLessThan(Money.valueOf(minPrice, code)));
    }

    public static Constraint<BigDecimal, Defect> cpmNotGreaterThan(BigDecimal maxPrice, CurrencyCode code) {
        // Money.valueOf кинет NPE на null, поэтому используем версию метода с Supplier
        return fromPredicateAndDefectSupplier(v ->  maxPrice == null || v.compareTo(maxPrice) <= 0,
                () -> MoneyDefects.invalidValueCpmNotGreaterThan(Money.valueOf(maxPrice, code)));
    }

    public static <T extends ModelWithId> Constraint<ModelChanges<T>, Defect> modelTypeSupportsSuitableInterface(
            Set<Long> supportSuitableInterfaceIds, Defect defect) {
        return Constraint.fromPredicate(modelChanges ->
                supportSuitableInterfaceIds.contains(modelChanges.getId()), defect);
    }
}
