package ru.yandex.direct.core.validation.constraints;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.mobilecontent.container.MobileAppStoreUrl;
import ru.yandex.direct.core.entity.mobilecontent.model.OsType;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.emptySet;
import static java.util.Objects.nonNull;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.appStoreIsNotSupported;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.invalidAppStoreUrl;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.invalidMinOsVersion;
import static ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser.GOOGLE_PLAY_DOMAIN;
import static ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser.ITUNES_APP_STORE_DOMAIN;
import static ru.yandex.direct.core.entity.mobilecontent.util.MobileAppStoreUrlParser.ITUNES_APP_STORE_OLD_DOMAIN;

@ParametersAreNonnullByDefault
public class MobileContentConstraints {
    /**
     * N.B.: аналогичные структуры есть и в Perl (protected/MobileContent.pm). Поправил в Java - поправь и в Perl.
     */
    public static final Map<OsType, Set<String>> OS_VERSIONS = ImmutableMap.of(
            OsType.ANDROID, ImmutableSet.of(
                    "1.0", "1.1", "1.5", "1.6",
                    "2.0", "2.1", "2.2", "2.3",
                    "3.0", "3.1", "3.2",
                    "4.0", "4.1", "4.2", "4.3", "4.4",
                    "5.0", "5.1",
                    "6.0",
                    "7.0", "7.1",
                    "8.0", "8.1",
                    "9.0",
                    "10.0",
                    "11.0"),
            OsType.IOS, ImmutableSet.of(
                    "1.0",
                    "2.0",
                    "3.0", "3.1",
                    "4.0", "4.1", "4.2",
                    "5.0", "5.1",
                    "6.0", "6.1",
                    "7.0", "7.1",
                    "8.0", "8.1", "8.2", "8.3", "8.4",
                    "9.0", "9.1", "9.2", "9.3",
                    "10.0", "10.1", "10.2", "10.3",
                    "11.0", "11.1", "11.2", "11.3", "11.4",
                    "12.0", "12.1", "12.2", "12.3", "12.4",
                    "13.0", "13.1", "13.2", "13.3", "13.4", "13.5", "13.6", "13.7",
                    "14.0", "14.1")
    );


    private static final List<String> ALLOWED_STORE_DOMAINS =
            ImmutableList.of(GOOGLE_PLAY_DOMAIN, ITUNES_APP_STORE_DOMAIN, ITUNES_APP_STORE_OLD_DOMAIN);

    private MobileContentConstraints() {
    }

    public static Constraint<String, Defect> validAppStoreDomain() {
        return Constraint.fromPredicate(MobileContentConstraints::isValidAppStoreDomain,
                appStoreIsNotSupported(ALLOWED_STORE_DOMAINS));
    }

    public static boolean isValidAppStoreDomain(String url) {
        try {
            String domain = new URI(url).getHost();
            return ALLOWED_STORE_DOMAINS.contains(domain);
        } catch (URISyntaxException e) {
            return false;
        }
    }

    public static Constraint<String, Defect> validAppStoreUrl(@Nullable MobileAppStoreUrl parsedUrl) {
        return Constraint.fromPredicate(url -> nonNull(parsedUrl), invalidAppStoreUrl());
    }

    public static Constraint<String, Defect> validOsVersion(@Nullable MobileAppStoreUrl parsedUrl) {
        return Constraint.fromPredicate(osVersion -> parsedUrl == null ||
                        OS_VERSIONS.getOrDefault(parsedUrl.getOsType(), emptySet()).contains(osVersion),
                invalidMinOsVersion());
    }
}
