package ru.yandex.direct.core.validation.defects.params;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Параметры для DefectId, когда ошибка касается какой-то коллекции и затрагивает
 * только часть элементов.
 * <p>
 * Пример: где-то есть валидация, что в коллекции все строки должны соответствовать
 * регулярному выражению. Валидатор, если обнаруживает, что часть этих строк
 * этому выражению не соответствует, порождает для Defect этот объект
 * с полем subset, в котором только строки, которые ему не соответствуют.
 */
@ParametersAreNonnullByDefault
public class CollectionSubsetDefectParams<T> {
    private final List<T> subset;

    private CollectionSubsetDefectParams(Collection<T> subset) {
        this.subset = new ArrayList<>(subset);
    }

    public static <T> CollectionSubsetDefectParams<T> of(@Nullable Collection<T> subset) {
        return new CollectionSubsetDefectParams<>(checkNotNull(subset));
    }

    public List<T> getSubset() {
        return subset;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        CollectionSubsetDefectParams<?> that = (CollectionSubsetDefectParams<?>) o;

        return subset.equals(that.subset);
    }

    @Override
    public int hashCode() {
        return subset.hashCode();
    }

    @Override
    public String toString() {
        return "CollectionSubsetDefectParams{" +
                "subset=" + subset +
                '}';
    }
}
