package ru.yandex.direct.core.validation.validators;

import java.util.List;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.campaign.model.InternalCampaignWithImpressionRate;
import ru.yandex.direct.core.entity.campaign.model.RfCloseByClickType;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.internalads.service.validation.defects.InternalAdDefects.requiredImpressionRateDueToRfCloseByClick;
import static ru.yandex.direct.core.validation.defects.Defects.requiredImpressionRateDueToAdsHasCloseCounter;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;

/**
 * Валидирует значения и взаимосвязь полей про частоту показа для внутренней рекламы
 * (impressionRate в кампании,rf/rfReset на группе)
 */
public class ModelWithInternalImpressionRateValidator<M extends Model> implements Validator<M, Defect> {

    private final boolean useValidatorForAdGroup;
    private final ModelProperty<? super M, Integer> propertyImpressionRateCount;
    private final ModelProperty<? super M, Integer> propertyImpressionRateIntervalDays;
    private final ModelProperty<? super M, Integer> propertyMaxClicksCount;
    private final ModelProperty<? super M, Integer> propertyMaxClicksPeriod;
    private final ModelProperty<? super M, Integer> propertyMaxStopsCount;
    private final ModelProperty<? super M, Integer> propertyMaxStopsPeriod;
    private final RfCloseByClickType rfCloseByClickType;
    private final List<Long> bannerWithCloseCounterVarIds;

    public ModelWithInternalImpressionRateValidator(ModelProperty<? super M, Integer> propertyImpressionRateCount,
                                                    ModelProperty<? super M, Integer> propertyImpressionRateIntervalDays,
                                                    ModelProperty<? super M, Integer> propertyMaxClicksCount,
                                                    ModelProperty<? super M, Integer> propertyMaxClicksPeriod,
                                                    ModelProperty<? super M, Integer> propertyMaxStopsCount,
                                                    ModelProperty<? super M, Integer> propertyMaxStopsPeriod,
                                                    @Nullable RfCloseByClickType rfCloseByClickType,
                                                    List<Long> bannerWithCloseCounterVarIds) {
        checkArgument(propertyImpressionRateCount == InternalAdGroup.RF
                        || propertyImpressionRateCount == InternalCampaignWithImpressionRate.IMPRESSION_RATE_COUNT,
                "unexpected propertyImpressionRateCount %s", propertyImpressionRateCount);
        this.useValidatorForAdGroup = propertyImpressionRateCount == InternalAdGroup.RF;

        this.propertyImpressionRateCount = propertyImpressionRateCount;
        this.propertyImpressionRateIntervalDays = propertyImpressionRateIntervalDays;
        this.propertyMaxClicksCount = propertyMaxClicksCount;
        this.propertyMaxClicksPeriod = propertyMaxClicksPeriod;
        this.propertyMaxStopsCount = propertyMaxStopsCount;
        this.propertyMaxStopsPeriod = propertyMaxStopsPeriod;
        this.rfCloseByClickType = rfCloseByClickType;
        this.bannerWithCloseCounterVarIds = bannerWithCloseCounterVarIds;
    }

    @Override
    public ValidationResult<M, Defect> apply(M model) {
        ModelItemValidationBuilder<M> vb = ModelItemValidationBuilder.of(model);

        vb.check(ifHasIntervalDaysCountShouldBeFilled());
        vb.check(ifCountValueAndPeriodValueAreTogetherFilled(propertyMaxClicksCount, propertyMaxClicksPeriod));
        vb.check(ifCountValueAndPeriodValueAreTogetherFilled(propertyMaxStopsCount, propertyMaxStopsPeriod));

        vb.item(propertyImpressionRateCount)
                .check(greaterThan(0))
                .check(notGreaterThan(CampaignConstants.MAX_IMPRESSION_RATE_COUNT_BANANA))
                .check(impressionRateNoAllowedToDeleteIfHasAdsWithCloseCounterVariable(), When.isValid())
                .check(impressionRateRequiredIfHasRfCloseByClick(), When.isValid());
        vb.item(propertyImpressionRateIntervalDays)
                .check(greaterThan(0))
                .check(notGreaterThan(CampaignConstants.MAX_IMPRESSION_RATE_INTERVAL_DAYS_FOR_INTERNAL_CAMPAIGNS));

        vb.item(propertyMaxClicksCount)
                .check(greaterThan(0))
                .check(notGreaterThan(CampaignConstants.MAX_IMPRESSION_RATE_COUNT_WITHOUT_FEATURE));
        vb.item(propertyMaxStopsCount)
                .check(greaterThan(0))
                .check(notGreaterThan(CampaignConstants.MAX_IMPRESSION_RATE_COUNT_WITHOUT_FEATURE));
        vb.item(propertyMaxClicksPeriod)
                .check(greaterThan(0))
                .check(notGreaterThan(CampaignConstants.MAX_IMPRESSION_RATE_INTERVAL_DAYS_FOR_INTERNAL_CAMPAIGNS_IN_SECONDS),
                        When.valueIs(v -> v != null && v != CampaignConstants.MAX_CLICKS_AND_STOPS_PERIOD_WHOLE_CAMPAIGN_VALUE));
        vb.item(propertyMaxStopsPeriod)
                .check(greaterThan(0))
                .check(notGreaterThan(CampaignConstants.MAX_IMPRESSION_RATE_INTERVAL_DAYS_FOR_INTERNAL_CAMPAIGNS_IN_SECONDS),
                        When.valueIs(v -> v != null && v != CampaignConstants.MAX_CLICKS_AND_STOPS_PERIOD_WHOLE_CAMPAIGN_VALUE));


        return vb.getResult();
    }

    /**
     * Если есть количество дней, то должно быть количество показов
     */
    private Constraint<M, Defect> ifHasIntervalDaysCountShouldBeFilled() {
        return Constraint.fromPredicate(
                model -> isImpressionRateCountFilledOnFillingImpressionRate(
                        propertyImpressionRateCount.get(model),
                        propertyImpressionRateIntervalDays.get(model)),
                inconsistentState());
    }

    private static boolean isImpressionRateCountFilledOnFillingImpressionRate(
            @Nullable Integer impressionRateCount, @Nullable Integer impressionRateIntervalDays) {
        return impressionRateCount != null || impressionRateIntervalDays == null;
    }

    private Constraint<M, Defect> ifCountValueAndPeriodValueAreTogetherFilled(ModelProperty<? super M, Integer> propertyCountValue,
                                                                              ModelProperty<? super M, Integer> propertyPeriodValue) {
        return Constraint.fromPredicate(
                model -> propertyCountValue.get(model) == null && propertyPeriodValue.get(model) == null
                        || propertyCountValue.get(model) != null && propertyPeriodValue.get(model) != null,
                inconsistentState());
    }

    /**
     * Проверяет, что при удалении Ограничения частоты показов не должно быть баннеров со Счетчикам закрытия
     */
    private Constraint<Integer, Defect> impressionRateNoAllowedToDeleteIfHasAdsWithCloseCounterVariable() {
        return Constraint.fromPredicateOfNullable(
                impressionRateCount -> !(impressionRateCount == null && !bannerWithCloseCounterVarIds.isEmpty()),
                requiredImpressionRateDueToAdsHasCloseCounter(bannerWithCloseCounterVarIds));
    }

    /**
     * Проверяет, что Ограничение частоты показов проставленно если есть Закрытие по клику
     */
    private Constraint<Integer, Defect> impressionRateRequiredIfHasRfCloseByClick() {
        return Constraint.fromPredicateOfNullable(
                impressionRateCount -> !(impressionRateCount == null && hasRfCloseByClickForValidatedObject()),
                requiredImpressionRateDueToRfCloseByClick());
    }

    /**
     * Проверяет есть ли Закрытие по клику для валидируемого объекта Кампания/Группа
     */
    private boolean hasRfCloseByClickForValidatedObject() {
        return (rfCloseByClickType == RfCloseByClickType.ADGROUP && useValidatorForAdGroup)
                || (rfCloseByClickType == RfCloseByClickType.CAMPAIGN && !useValidatorForAdGroup);
    }

}
