package ru.yandex.direct.core.entity.keyword.processing;

import java.util.Collection;
import java.util.List;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.model.Keyword;
import ru.yandex.direct.libs.keywordutils.parser.KeywordParser;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.assertThat;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeDuplicates;
import static ru.yandex.direct.core.entity.keyword.processing.MinusKeywordsDeduplicator.removeDuplicatesStr;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@RunWith(Parameterized.class)
public class MinusKeywordDeduplicatorTest {

    @Parameterized.Parameter
    public List<String> minusKeywordsStr;

    @Parameterized.Parameter(1)
    public List<String> expectedUniqueMinusKeywordsStr;

    @Parameterized.Parameters(name = "входные минус-фразы: {0}, ожидаемые минус-фразы: {1}")
    public static Collection<Object[]> parameters() {
        return asList(new Object[][]{
                // а вообще работает?
                {singletonList("конь"), singletonList("конь")}, // 1 лемма
                {singletonList("уха"), singletonList("уха")}, // 2 леммы

                // одно слово без операторов + учет регистра + учет точки в конце
                {
                        asList("конь", "конь"),
                        singletonList("конь")
                },
                {
                        asList("конь", "Конь"),
                        singletonList("конь")
                },
                {
                        asList("конь", "конь."),
                        singletonList("конь")
                },
                {
                        asList("конь", "Конь."),
                        singletonList("конь")
                },
                {
                        asList("Конь", "конь"),
                        singletonList("Конь")
                },
                {
                        asList("конь.", "конь"),
                        singletonList("конь.")
                },
                {
                        asList("Конь.", "конь"),
                        singletonList("Конь.")
                },
                {
                        asList("коня", "конь"),
                        singletonList("коня")
                },
                {
                        asList("Коня", "конь"),
                        singletonList("Коня")
                },
                {
                        asList("коня", "Конь"),
                        singletonList("коня")
                },
                {
                        asList("коня", "вася"),
                        asList("коня", "вася")
                },
                {
                        asList("идти", "коня", "велосипед", "конь", "трамвай"),
                        asList("идти", "коня", "велосипед", "трамвай")
                },
                {
                        asList("идти", "коня", "велосипед", "конь", "трамвай", "коню"),
                        asList("идти", "коня", "велосипед", "трамвай")
                },

                // отдельно проверим эквивалентность "ё" и "е" + учет регистра
                {
                        asList("ёлка", "елка"),
                        singletonList("елка")
                },
                {
                        asList("ёлка", "Елка"),
                        singletonList("елка")
                },
                {
                        asList("елка", "Ёлка"),
                        singletonList("елка")
                },

                // только "уха" может включать "ухо", но не наоборот
                {
                        asList("уха", "ухо"),   // второе слово лишнее, т.к. уха - это суп и ухо
                        singletonList("уха")
                },
                {
                        asList("ухо", "уха"),   // первое слово лишнее, т.к. уха - это суп и ухо
                        singletonList("уха")
                },
                {
                        asList("ухо", "Уха."),   // регистр и точка не должны влиять
                        singletonList("Уха.")
                },
                {
                        asList("ухо", "конь", "уха"),
                        asList("конь", "уха")
                },
                {
                        asList("ухо", "конь", "уха", "ухе", "трамвай", "уши"),
                        asList("конь", "уха", "трамвай")
                },

                // одно слово с плюсом (плюс игнорируется)
                {
                        asList("конь", "+конь"),
                        singletonList("конь")
                },
                {
                        asList("+конь", "конь"),
                        singletonList("+конь")
                },
                {
                        asList("+Конь", "конь"),
                        singletonList("+Конь")
                },
                {
                        asList("+коня", "конь"),
                        singletonList("+коня")
                },
                {
                        asList("+коня", "Конь"),
                        singletonList("+коня")
                },

                // одно слово с фиксирующим оператором
                {
                        asList("конь", "!конь"),
                        singletonList("конь")
                },
                {
                        asList("конь", "!Конь"),
                        singletonList("конь")
                },
                {
                        asList("!конь", "конь"),
                        singletonList("конь")
                },
                {
                        asList("!конь", "Конь"),
                        singletonList("Конь")
                },
                {
                        asList("!конь", "Конь."),
                        singletonList("Конь.")
                },
                {
                        asList("!коня", "конь"),
                        singletonList("конь")
                },
                {
                        asList("конь", "!вася"),
                        asList("конь", "!вася")
                },
                {
                        asList("коню", "!коня"),
                        singletonList("коню")
                },
                {
                        asList("!коню", "коня"),
                        singletonList("коня")
                },
                {
                        asList("уши", "!уха"),  // уха - это суп и ухо
                        singletonList("уши")
                },
                {
                        asList("!уши", "уха"),  // уха - это суп и ухо
                        singletonList("уха")
                },
                {
                        asList("!кия", "киев"),  // есть одна пересекающаяся лемма
                        singletonList("киев")
                },
                {
                        asList("кия", "!киев"),  // есть одна пересекающаяся лемма
                        singletonList("кия")
                },
                {
                        asList("!уши", "коня", "уха", "!коню"),  // уха - это суп и ухо
                        asList("коня", "уха")
                },
                {
                        asList("!конь", "!конь"),
                        singletonList("!конь")
                },
                {
                        asList("!конь", "!коня"),
                        asList("!конь", "!коня")
                },
                {
                        asList("!конь", "коню", "!конь", "!кони"),
                        singletonList("коню")
                },

                // одно слово в квадратных скобках
                {
                        asList("[конь]", "[конь]"),
                        singletonList("[конь]")
                },
                {
                        asList("[конь]", "[Конь.]"),
                        singletonList("[конь]")
                },
                {
                        asList("[конь]", "[коню]"),
                        singletonList("[конь]")
                },
                {
                        asList("[Конь.]", "[коню]"),
                        singletonList("[Конь.]")
                },
                {
                        asList("[конь]", "[вася]"),
                        asList("[конь]", "[вася]")
                },
                // только "уха" может включать "ухо", но не наоборот
                {
                        asList("[ухо]", "[уха]"),
                        singletonList("[уха]")
                },
                {
                        asList("[уха]", "[ухо]"),
                        singletonList("[уха]")
                },
                {
                        asList("[!ухо]", "[ухо]"),
                        singletonList("[ухо]")
                },
                {
                        asList("[!ухо]", "[уши]"),
                        singletonList("[уши]")
                },
                {
                        asList("[!ухо]", "[уха]"),
                        singletonList("[уха]")
                },
                {
                        asList("[ухо]", "[!уха]"),
                        singletonList("[ухо]")
                },
                {
                        asList("[!ухо]", "[!ухо]"),
                        singletonList("[!ухо]")
                },
                {
                        asList("[!ухо]", "[!уха]"),
                        asList("[!ухо]", "[!уха]")
                },
                {
                        asList("[!ухо]", "карты", "[ухо]", "паровоз"),
                        asList("карты", "[ухо]", "паровоз")
                },

                // одно слово в кавычках (дублируют кейсы для одного слова в квадратных скобках)
                {
                        asList("\"конь\"", "\"конь\""),
                        singletonList("\"конь\"")
                },
                {
                        asList("\"конь\"", "\"Конь.\""),
                        singletonList("\"конь\"")
                },
                {
                        asList("\"конь\"", "\"коню\""),
                        singletonList("\"конь\"")
                },
                {
                        asList("\"конь\"", "\"Коню\""),
                        singletonList("\"конь\"")
                },
                {
                        asList("\"Конь.\"", "\"коню\""),
                        singletonList("\"Конь.\"")
                },
                {
                        asList("\"конь\"", "\"вася\""),
                        asList("\"конь\"", "\"вася\"")
                },
                // только "уха" может включать "ухо", но не наоборот
                {
                        asList("\"ухо\"", "\"уха\""),
                        singletonList("\"уха\"")
                },
                {
                        asList("\"уха\"", "\"ухо\""),
                        singletonList("\"уха\"")
                },
                {
                        asList("\"!ухо\"", "\"ухо\""),
                        singletonList("\"ухо\"")
                },
                {
                        asList("\"!ухо\"", "\"уши\""),
                        singletonList("\"уши\"")
                },
                {
                        asList("\"!ухо\"", "\"уха\""),
                        singletonList("\"уха\"")
                },
                {
                        asList("\"ухо\"", "\"!уха\""),
                        singletonList("\"ухо\"")
                },
                {
                        asList("\"!ухо\"", "\"!ухо\""),
                        singletonList("\"!ухо\"")
                },
                {
                        asList("\"!ухо\"", "\"!уха\""),
                        asList("\"!ухо\"", "\"!уха\"")
                },
                {
                        asList("\"!ухо\"", "карты", "\"ухо\"", "паровоз"),
                        asList("карты", "\"ухо\"", "паровоз")
                },

                // разное количество слов
                {
                        asList("конь конь", "конь"),
                        asList("конь конь", "конь")
                },
                {
                        asList("конь конь", "!конь"),
                        asList("конь конь", "!конь")
                },
                {
                        asList("!конь !конь", "!конь"),
                        asList("!конь !конь", "!конь")
                },
                {
                        asList("!конь !конь", "конь"),
                        asList("!конь !конь", "конь")
                },
                {
                        asList("[конь конь]", "[конь]"),
                        asList("[конь конь]", "[конь]")
                },
                {
                        asList("\"конь конь\"", "\"конь\""),
                        asList("\"конь конь\"", "\"конь\"")
                },

                // одно слово в квадратных скобках, а второе - нет
                {
                        asList("конь", "[конь]"),
                        asList("конь", "[конь]")
                },
                {
                        asList("конь [конь]", "конь"),
                        asList("конь [конь]", "конь")
                },

                // одно слово в кавычках, а второе - нет
                {
                        asList("конь", "\"конь\""),
                        singletonList("конь")
                },
                {
                        asList("коню", "\"коня\""),
                        singletonList("коню")
                },

                // два слова без квадратных скобок и кавычек
                {
                        asList("вакуумный конь", "вакуумному коню"),
                        singletonList("вакуумный конь")
                },
                {
                        asList("вакуумному коню", "вакуумный конь"),
                        singletonList("вакуумному коню")
                },
                // только "вкусная уха" может включать "вкусные уши", но не наоборот
                {
                        asList("вкусная уха", "вкусные уши"),
                        singletonList("вкусная уха")
                },
                {
                        asList("вкусные уши", "уха вкусная"),
                        singletonList("уха вкусная")
                },
                {
                        asList("вакуумному коню", "вакуумный +конь"),
                        singletonList("вакуумному коню")
                },
                {
                        asList("конь вакуумный", "вакуумному коню"),
                        singletonList("конь вакуумный")
                },
                {
                        asList("вакуумный конь", "вакуумному !коню"),
                        singletonList("вакуумный конь")
                },
                {
                        asList("!вакуумному !коню", "вакуумный конь"),
                        singletonList("вакуумный конь")
                },
                {
                        asList("вакуумный !конь", "вакуумному !коню"),
                        asList("вакуумный !конь", "вакуумному !коню")
                },
                {
                        asList("!вакуумный конь", "вакуумному !коню"),
                        asList("!вакуумный конь", "вакуумному !коню")
                },
                {
                        asList("!вакуумный конь", "вакуумный", "вакуумному коню", "вакуумный !конь"),
                        asList("вакуумный", "вакуумному коню")
                },

                // два слова в квадратных скобках
                {
                        asList("[вакуумный конь]", "[вакуумному коню]"),
                        singletonList("[вакуумный конь]")
                },
                {
                        asList("[вакуумному коню]", "[вакуумный конь]"),
                        singletonList("[вакуумному коню]")
                },
                {
                        asList("[вакуумному коню]", "[вакуумный +конь]"),
                        singletonList("[вакуумному коню]")
                },
                {
                        asList("[вакуумный конь]", "[конь вакуумный]"),
                        asList("[вакуумный конь]", "[конь вакуумный]")
                },
                {
                        asList("[вакуумный !конь]", "[вакуумному коню]"),
                        singletonList("[вакуумному коню]")
                },
                {
                        asList("[вакуумный !конь]", "[!вакуумный конь]"),
                        asList("[вакуумный !конь]", "[!вакуумный конь]")
                },
                {
                        asList("[вакуумный !конь]", "[вакуумному коню]", "[!вакуумный !конь]", "[вакуумный]"),
                        asList("[вакуумному коню]", "[вакуумный]")
                },

                // два слова в кавычках
                {
                        asList("\"вакуумный конь\"", "\"вакуумному коню\""),
                        singletonList("\"вакуумный конь\"")
                },
                {
                        asList("\"вакуумному коню\"", "\"вакуумный конь\""),
                        singletonList("\"вакуумному коню\"")
                },
                // только "вкусная уха" может включать "вкусные уши", но не наоборот
                {
                        asList("\"вкусная уха\"", "\"вкусные уши\""),
                        singletonList("\"вкусная уха\"")
                },
                {
                        asList("\"вкусные уши\"", "\"уха вкусная\""),
                        singletonList("\"уха вкусная\"")
                },
                {
                        asList("\"вакуумному коню\"", "\"вакуумный +конь\""),
                        singletonList("\"вакуумному коню\"")
                },
                {
                        asList("\"вакуумный конь\"", "\"конь вакуумный\""),
                        singletonList("\"вакуумный конь\"")
                },
                {
                        asList("\"вакуумный !конь\"", "\"вакуумному коню\""),
                        singletonList("\"вакуумному коню\"")
                },
                {
                        asList("\"вакуумный !конь\"", "\"коню вакуумному\""),
                        singletonList("\"коню вакуумному\"")
                },
                {
                        asList("\"вакуумный !конь\"", "\"!вакуумный конь\""),
                        asList("\"вакуумный !конь\"", "\"!вакуумный конь\"")
                },
                {
                        asList("\"вакуумный !конь\"", "\"вакуумному коню\"", "\"!вакуумный !конь\"", "\"вакуумный\""),
                        asList("\"вакуумному коню\"", "\"вакуумный\"")
                },

                // отдельные слова + квадратные скобки
                {
                        asList("[вакуумному коню] респект", "респект [вакуумный конь]"),
                        singletonList("[вакуумному коню] респект")
                },
                {
                        asList("[вакуумный конь] вася", "[вакуумный конь] респект"),
                        asList("[вакуумный конь] вася", "[вакуумный конь] респект")
                },
                {
                        asList("[вакуумному коню] респект", "[вакуумный конь респект]"),
                        asList("[вакуумному коню] респект", "[вакуумный конь респект]")
                },
                {
                        asList("[вакуумному коню] +в !зубы +не смотрят", "[вакуумный !конь] смотреть не +в !зубы"),
                        singletonList("[вакуумному коню] +в !зубы +не смотрят")
                },
                {
                        asList("[вакуумному коню] +в !зубы +не смотрят", "смотреть !зубы [вакуумный конь] +не +в"),
                        singletonList("[вакуумному коню] +в !зубы +не смотрят")
                },
                {
                        asList("[вакуумному коню] +в !зубы +не смотрят", "!смотреть !зубы не +в [вакуумный конь]"),
                        singletonList("[вакуумному коню] +в !зубы +не смотрят")
                },
                {
                        asList("[вакуумному коню] +в !зубы +не !смотрят", "!зубы смотреть [вакуумный конь] +не +в"),
                        singletonList("!зубы смотреть [вакуумный конь] +не +в")
                },
                {
                        asList("[вакуумному коню] +в зубы +не смотрят", "зубы [вакуумный !конь] !смотреть не +в"),
                        singletonList("[вакуумному коню] +в зубы +не смотрят")
                },
                {
                        asList("[вакуумному !коню] +в зубы +не смотрят", "[вакуумному коню] +в !зуб +не смотрят"),
                        asList("[вакуумному !коню] +в зубы +не смотрят", "[вакуумному коню] +в !зуб +не смотрят")
                },
                {
                        asList("[вакуумному коню] +в зубы +не [смотрят]", "[вакуумному коню] +в !зуб +не смотрят"),
                        asList("[вакуумному коню] +в зубы +не [смотрят]", "[вакуумному коню] +в !зуб +не смотрят")
                },

                // две группы слов в квадратных скобках
                {
                        asList("[вакуумному коню] +в зубы [не смотрят]", "[вакуумному коню] [не смотреть] +в зубы"),
                        singletonList("[вакуумному коню] +в зубы [не смотрят]")
                },
                {
                        asList("[не смотреть] +в зубы [вакуумному коню]", "+в [вакуумному коню] зубы [не смотрят]"),
                        singletonList("[не смотреть] +в зубы [вакуумному коню]")
                },

                // отдельные слова в кавычках + квадратные скобки
                // (дублирует аналогичные кейсы с квадратными скобками без кавычек)
                {
                        asList("\"[вакуумному коню] респект\"", "\"респект [вакуумный конь]\""),
                        singletonList("\"[вакуумному коню] респект\"")
                },
                {
                        asList("\"[вакуумный конь] вася\"", "\"[вакуумный конь] респект\""),
                        asList("\"[вакуумный конь] вася\"", "\"[вакуумный конь] респект\"")
                },
                {
                        asList("\"[вакуумному коню] респект\"", "\"[вакуумный конь респект]\""),
                        asList("\"[вакуумному коню] респект\"", "\"[вакуумный конь респект]\"")
                },
                {
                        asList("\"[вакуумному коню] +в !зубы +не смотрят\"",
                                "\"[вакуумный !конь] смотреть не +в !зубы\""),
                        singletonList("\"[вакуумному коню] +в !зубы +не смотрят\"")
                },
                {
                        asList("\"[вакуумному коню] +в !зубы +не смотрят\"",
                                "\"смотреть !зубы [вакуумный конь] +не +в\""),
                        singletonList("\"[вакуумному коню] +в !зубы +не смотрят\"")
                },
                {
                        asList("\"[вакуумному коню] +в !зубы +не смотрят\"",
                                "\"!смотреть !зубы не +в [вакуумный конь]\""),
                        singletonList("\"[вакуумному коню] +в !зубы +не смотрят\"")
                },
                {
                        asList("\"[вакуумному коню] +в !зубы +не !смотрят\"",
                                "\"!зубы смотреть [вакуумный конь] +не +в\""),
                        singletonList("\"!зубы смотреть [вакуумный конь] +не +в\"")
                },
                {
                        asList("\"[вакуумному коню] +в зубы +не смотрят\"",
                                "\"зубы [вакуумный !конь] !смотреть не +в\""),
                        singletonList("\"[вакуумному коню] +в зубы +не смотрят\"")
                },
                {
                        asList("\"[вакуумному !коню] +в зубы +не смотрят\"",
                                "\"[вакуумному коню] +в !зуб +не смотрят\""),
                        asList("\"[вакуумному !коню] +в зубы +не смотрят\"",
                                "\"[вакуумному коню] +в !зуб +не смотрят\"")
                },
                {
                        asList("\"[вакуумному коню] +в зубы +не [смотрят]\"",
                                "\"[вакуумному коню] +в !зуб +не смотрят\""),
                        asList("\"[вакуумному коню] +в зубы +не [смотрят]\"",
                                "\"[вакуумному коню] +в !зуб +не смотрят\"")
                },

                // две группы слов в кавычках и квадратных скобках
                {
                        asList("\"[вакуумному коню] +в зубы [не смотрят]\"",
                                "\"[вакуумному коню] [не смотреть] +в зубы\""),
                        singletonList("\"[вакуумному коню] +в зубы [не смотрят]\"")
                },
                {
                        asList("\"[не смотреть] +в зубы [вакуумному коню]\"",
                                "\"+в [вакуумному коню] зубы [не смотрят]\""),
                        singletonList("\"[не смотреть] +в зубы [вакуумному коню]\"")
                },

                // (слова в кавычках + квадратные скобки) сравниваются со (словами без кавычек + квадратные скобки)
                // только фраза в кавычках может дублировать фразу без кавычек, но не наоборот
                {
                        asList("\"[вакуумному коню] респект\"", "респект [вакуумный конь]"),
                        singletonList("респект [вакуумный конь]")
                },
                {
                        asList("респект [вакуумный конь]", "\"[вакуумному коню] респект\""),
                        singletonList("респект [вакуумный конь]")
                },
                {
                        asList("вася [вакуумный конь]", "\"респект [вакуумный конь]\""),
                        asList("вася [вакуумный конь]", "\"респект [вакуумный конь]\"")
                },
                // здесь у слова "уха" больше лемм, но оно в кавычках, так что не судьба
                {
                        asList("ухо [вакуумный конь]", "\"уха [вакуумный конь]\""),
                        asList("ухо [вакуумный конь]", "\"уха [вакуумный конь]\"")
                },
                // слова с символами из разного алфавита
                {
                        asList("конь", "кoнь"),                         // второе слово с англ. 'o'
                        asList("конь", "кoнь")
                },
                {
                        asList("кocа", "кocа"),                         // оба слова с англ. 'o'
                        singletonList("кocа")
                },
                {
                        asList("коса", "коса", "кocа", "кoса", "коcа"), // повторяется только 2 первых слова на русском
                        asList("коса", "кocа", "кoса", "коcа")          // остальные с чередованием англ. 'o' и 'c'
                },
                {
                        asList("коса", "кocа", "кocа"),                 // повторяются два последних слова с англ. 'o'
                        asList("коса", "кocа")
                },
                {
                        asList("коня", "конь", "кoнь"),                 // последнее слово с англ. 'o'
                        asList("коня", "кoнь")
                },
                {
                        asList("!коню", "коня", "кoнь"),                // последнее слово с англ. 'o'
                        asList("коня", "кoнь")
                },
                {
                        asList("[конь]", "[кoнь]"),                     // последнее слово с англ. 'o'
                        asList("[конь]", "[кoнь]")
                },
                {
                        asList("[Конь.]", "[кoнь]"),                    // последнее слово с англ. 'o'
                        asList("[Конь.]", "[кoнь]")
                },
                {
                        asList("[!ухо]", "[уха]", "[!уxа]"),            // последнее слово с англ. 'x'
                        asList("[уха]", "[!уxа]")
                },
                {
                        asList("конь", "\"конь\"", "\"кoнь\""),         // последнее слово с англ. 'o'
                        asList("конь", "\"кoнь\"")
                },
                {
                        asList("\"!ухо\"", "\"уха\"", "\"уxа\""),       // последнее слово с англ. 'x'
                        asList("\"уха\"", "\"уxа\"")
                },
                {
                        asList("\"вакуумный конь\"", "\"вакуумный кoнь\""), // последнее слово с англ. 'o'
                        asList("\"вакуумный конь\"", "\"вакуумный кoнь\"")
                },
                {
                        asList("\"вакуумный !конь\"",
                                "\"коню вакуумному\"",
                                "\"вакуумный кoнь\""),                  // последнее слово с англ. 'o'
                        asList("\"коню вакуумному\"", "\"вакуумный кoнь\"")
                },
                {
                        asList("[вакуумному коню] респект",
                                "респект [вакуумный конь]",
                                "респект [вакуумный кoнь]"),            // последнее слово с англ. 'o'
                        asList("[вакуумному коню] респект", "респект [вакуумный кoнь]")
                },
                {
                        asList("\"[вакуумному коню] респект\"",
                                "респект [вакуумный конь]",
                                "респект [вакуумный кoнь]"),            // последнее слово с англ. 'o'
                        asList("респект [вакуумный конь]", "респект [вакуумный кoнь]")
                },
                {
                        asList("\"[вакуумному коню] +в !зубы +не смотрят\"",
                                "\"[вакуумный !конь] смотреть не +в !зубы\"",
                                "\"[вакуумный !кoнь] смотреть не +в !зубы\""),  // последнее слово с англ. 'o'
                        asList("\"[вакуумному коню] +в !зубы +не смотрят\"",
                                "\"[вакуумный !кoнь] смотреть не +в !зубы\"")
                },
                {
                        asList("конь конь конь конь конь конь конь",
                                "конь конь конь конь конь конь конь",
                                "конь",
                                "конь"),
                        asList("конь конь конь конь конь конь конь",
                                "конь")
                },
                // фразы длиннее 7 слов не дедуплицируются
                {
                        asList("конь конь конь конь конь конь конь конь",
                                "конь конь конь конь конь конь конь конь",
                                "конь",
                                "конь"),
                        asList("конь конь конь конь конь конь конь конь",
                                "конь конь конь конь конь конь конь конь",
                                "конь")
                }
        });
    }

    @Test
    public void removeDuplicates_WorksFine() {
        List<Keyword> minusKeywords = mapList(minusKeywordsStr, KeywordParser::parse);
        List<Keyword> actualUniqueMinusKeywords = removeDuplicates(new SingleKeywordsCache(), minusKeywords);

        List<String> actualUniqueMinusKeywordsStr = mapList(actualUniqueMinusKeywords, Keyword::toString);
        assertThat("полученные минус-фразы без дубликатов не соответствуют ожидаемым",
                actualUniqueMinusKeywordsStr, contains(expectedUniqueMinusKeywordsStr.toArray()));
    }

    @Test
    public void removeDuplicatesStr_WorksFine() {
        List<String> actualUniqueMinusKeywordsStr =
                removeDuplicatesStr(new SingleKeywordsCache(), minusKeywordsStr, t -> KeywordParser.parse(t));

        assertThat("полученные минус-фразы без дубликатов не соответствуют ожидаемым",
                actualUniqueMinusKeywordsStr, contains(expectedUniqueMinusKeywordsStr.toArray()));
    }
}
